!-----------------------------------------------------------------------------------------------------------------------------------------
! TITLE: ay_constants.f95
! AUTHOR: Alex Yuffa
! DATE WRITTEN: 08/07/04
! LAST REVISION: 07/09/05
! DESCRIPTION: Contains constant parameters that are used in the integral equations.
!-----------------------------------------------------------------------------------------------------------------------------------------

!-----------------------------------------------------------------------------------------------------------------------------------------
! NON-LOCAL VARIABLES USED:
! MODULE NAME:            VARIABLE NAME:      TYPE:                   COMMENT:
! ay_kind                 wp                  Integer, Parameter      Working precision
!-----------------------------------------------------------------------------------------------------------------------------------------

!-----------------------------------------------------------------------------------------------------------------------------------------
! LOCAL VARIABLES USED:
! VARIABLE NAME:          TYPE:                   COMMENT:
! ay_spw                  INTEGER, PARAMETER      Number of sample points per wavelength.
! ay_pi                   REAL, PARAMETER         The numerical value of pi.
! ay_k1                   REAL, PARAMETER         Wavenumber in Region 1.
! ay_k2                   REAL, PARAMETER         Wavenumber in Region 2.
! ay_lambda1              REAL, PARAMETER         Wavelength in Region 1.
! ay_R                    REAL, PARAMETER         Radius of a sphere or Radius of a disk in the x-y plane (z=0)
! ay_I                    COMPLEX, PARMETER       Sqrt(-1)
!-----------------------------------------------------------------------------------------------------------------------------------------

MODULE ay_constants
  USE ay_kind, ONLY : wp
  IMPLICIT NONE
! Sampling and Geometry 
  INTEGER, SAVE :: ay_spw 
  REAL(wp), SAVE :: ay_c
  REAL(wp), SAVE :: ay_R

! CONSTANTS
  REAL(wp), PARAMETER :: ay_pi =  3.14159265358979323846264338328_wp
  COMPLEX(wp), PARAMETER :: ay_I=(0.0_wp,1.0_wp)

! REGION 1 -- Enviroment: default is vacuum
  REAL(wp), PARAMETER :: lambda_1 = 2.0_wp*ay_pi
  REAL(wp), PARAMETER :: epsilon_1 = 1.0_wp !DON'T CHANGE 
  REAL(wp), PARAMETER :: mu_1 = 1.0_wp !DON'T CHANGE
  REAL(wp), PARAMETER :: ay_k1 = 1.0     !=2.0_wp*ay_pi/lambda_1  
  REAL(wp), PARAMETER :: ay_light = 1.0_wp !DON'T CHANGE
  REAL(wp), PARAMETER :: omega = ay_k1*ay_light
  REAL(wp), PARAMETER :: lambda_o = lambda_1
  REAL(wp), PARAMETER :: ay_ko = ay_k1


!FIXED REGION 2 = Scattered
  COMPLEX(wp), SAVE :: epsilon_2
  REAL(wp), SAVE :: mu_2
  COMPLEX(wp), SAVE :: ay_k2 
  

!FIXED RATION = scatterer/enviroment
COMPLEX(wp), SAVE :: ay_epsilon
REAL(wp),    SAVE :: ay_mu 

CONTAINS

  SUBROUTINE  ay_spw_write(pts)
    INTEGER, INTENT(IN) :: pts

    ay_spw = pts
  END SUBROUTINE ay_spw_write
!********************************************************************

SUBROUTINE ay_mu_write(mu_loc)
  REAL(wp), INTENT(IN) :: mu_loc

  mu_2 = mu_loc
END SUBROUTINE ay_mu_write
!********************************************************************

SUBROUTINE ay_epsilon_write(epsilon_re_loc,epsilon_im_loc)
  REAL(wp), INTENT(IN) :: epsilon_re_loc
  REAL(wp), INTENT(IN) :: epsilon_im_loc
  !Local Variables
  REAL(wp) :: epsilon_mag 
  REAL(wp) :: angle
  COMPLEX(wp) :: root_epsilon
  
  epsilon_2 = CMPLX(epsilon_re_loc,epsilon_im_loc) 

  !******** Warning Im[ay_k2]>= 0 because Exp[-I*omega*t] time factor is used
  ! Will use acos(x) because 0<=acos(x)<=Pi for |x|<=1 to find angle
  ! -infintiy < epsilon_re_loc < infinity and 0<= epsilon_im_loc < infinity
  epsilon_mag = SQRT(epsilon_re_loc**2 + epsilon_im_loc**2)
  angle = acos(epsilon_re_loc/epsilon_mag)
  root_epsilon =  SQRT(epsilon_mag)*Exp(ay_I*angle/2.0_wp)
  !******** End of Warning 

  ay_k2 = root_epsilon*SQRT(mu_2)*omega
  ay_epsilon = epsilon_2
  ay_mu = mu_2

END SUBROUTINE ay_epsilon_write
!*********************************************************************  

SUBROUTINE ay_R_write(rad)
  REAL(wp), INTENT(IN) :: rad
  ay_R = rad
  ay_c = 0.2_wp !2.0 !ay_R
END SUBROUTINE ay_R_write

END MODULE ay_constants
