!-----------------------------------------------------------------------------------------------------------------------------------------
! TITLE: ay_integration.f95
! AUTHOR: Alex Yuffa
! DATE WRITTEN: 08/08/04
! LAST REVISION: 04/17/05
! DESCRIPTION:  Contains helper functions for integration and sampling algorithms for integration.
!-----------------------------------------------------------------------------------------------------------------------------------------

!-----------------------------------------------------------------------------------------------------------------------------------------
! NON-LOCAL VARIABLES USED:
! MODULE NAME:            VARIABLE NAME:      TYPE:                   COMMENT:
! ay_kind                 wp                  INTEGER, PARAMETER      Working precision
! ay_constants            ay_pi               REAL, PARAMETER         Value of Pi
! ay_constants            ay_lambda1          REAL, PARAMETER         Wavelength in Region 1
! ay_constants            ay_R                REAL, PARAMETER         Radius of the sphere
! ay_constants            ay_spw              INTEGER                 Number of sample points per wavelength
! ay_constants            ay_spw_rho          INTEGER                 Number of sample points per wavelength in rho = ay_spw_rho*ay_spw
!-----------------------------------------------------------------------------------------------------------------------------------------

MODULE ay_integration
  USE ay_kind, ONLY : wp
  USE ay_constants, ONLY : ay_pi, ay_R, lambda_o, ay_spw, ay_epsilon, ay_c

  IMPLICIT NONE

CONTAINS

  FUNCTION ay_distance_cyl(x,xp)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: ay_write_obs_pt
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 08/08/04
    ! LAST REVISION: 
    ! DESCRIPTION: Computes distance between x and xp, where x is observation point and xp is source point in cylindrical coord. system.
    !              x(1) = rho, x(2) = theta, x(3) = z; xp(1) = rho prime, xp(2) = theta prime, xp(3) = z prime
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:       TYPE:               COMMENT:
    ! x                  IN            REAL, ARRAY         x observation point in cylindrical coord. system. x(1) = rho, x(2) = theta, 
    !                                                      x(3) = z
    ! xp                 IN            REAL, ARRAY         xp  source point in cylindrical coord. system. xp(1) = rho prime, 
    !                                                      xp(2) = theta prime, xp(3) = z prime
    ! ay_distance_cyl    OUT           REAL                Distance between x and xp
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x(:), xp(:)
    REAL(wp) :: ay_distance_cyl
 
    ay_distance_cyl = Sqrt( (x(3)-xp(3))**2 + x(1)**2 + xp(1)**2 - 2.0_wp*x(1)*xp(1)*Cos(x(2)-xp(2)) )
  END FUNCTION ay_distance_cyl
!*****************************************************************************************************************************************

  FUNCTION ay_n_rho_p()
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: ay_n_rho_p
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 08/19/04
    ! LAST REVISION: 12/29/04
    ! DESCRIPTION: 
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:       TYPE:               COMMENT:
    ! ay_n_rho_p         OUT           INTEEGER            Number of Discrete rho primes, i.e., rho_p(i), 0<=i<=ay_n_rho_p
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    INTEGER :: ay_n_rho_p

    ay_n_rho_p = ceiling(ay_pi*ay_spw*ay_R/(2.0_wp*lambda_o))
  END FUNCTION ay_n_rho_p
!*************************************************************************************************************************************    

  SUBROUTINE ay_sample_rho_p(rho_p)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: ay_sample_rho_p
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 08/08/04
    ! LAST REVISION: 12/29/04
    ! DESCRIPTION: Commutes discrete rhop primes for sampling.
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:            TYPE:           COMMENT:
    ! rho_p              OUT                REAL, ARRAY     Discrete rho primes, rho_p(i), 1<=i<= ay_n_rho_p
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(OUT) :: rho_p(:)
    INTEGER  :: i, n_rho_p
 
    n_rho_p = ay_n_rho_p()
 
    do i=0, n_rho_p       
       rho_p(i+1) = ay_R*Sin(ay_Pi*i/(2*n_rho_p)) ! rho_p(1), rho_p(2),...,rho_p(n+1)
    end do

    !Fixing the end points.
    rho_p(1) = epsilon(lambda_o)*1.0e-3_wp
    rho_p(n_rho_p+1) = rho_p(n_rho_p+1) - epsilon(lambda_o)*1.0e-3_wp
  END SUBROUTINE ay_sample_rho_p
!*****************************************************************************************************************************************

  FUNCTION ay_n_theta_p(rho_mid_p)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: ay_n_theta_p
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 08/08/04
    ! LAST REVISION: 11/28/04
    ! DESCRIPTION: 
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:            TYPE:           COMMENT:
    ! rho_p              OUT                REAL, ARRAY     Discrete rho primes, rho_p(i), 1<=i<= ay_n_rho_p
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: rho_mid_p 
    INTEGER  :: ay_n_theta_p

    ay_n_theta_p = ceiling(2.0_wp*ay_pi*ay_spw*rho_mid_p/lambda_o)
  END FUNCTION ay_n_theta_p

  SUBROUTINE ay_sample_theta_p(rho_mid_p, theta_p)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: ay_sample_theta_p
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 08/08/04
    ! LAST REVISION: 10/22/04
    ! DESCRIPTION: Determines sampling in theta prime.  WARNING: This subroutine MUST ALWAYS BE CALLED after ay_sample_rho_p subroutine.
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:            TYPE:           COMMENT:
    ! rho_mid_p          IN                 REAL            Average rho prime to be used
    ! theta_p            OUT                REAL, Array     Discrete Theta primes
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: rho_mid_p
    REAL(wp), INTENT(OUT) :: theta_p(:)
    INTEGER :: i, n_theta_p

    n_theta_p = ay_n_theta_p(rho_mid_p)

    do i=0, n_theta_p
       theta_p(i+1) = 2.0_wp*ay_pi*i/n_theta_p ! theta_p(1), theta_p(2),...,theta_p(n+1)
    end do
  END SUBROUTINE ay_sample_theta_p
!*****************************************************************************************************************************************

FUNCTION ay_tot_num_obs_pts()
  ! The count is for ONE surface, thus, true total = 2 * ay_tot_num_obs_pts
  IMPLICIT NONE
  INTEGER :: ay_tot_num_obs_pts
  INTEGER :: i, count_sample_pts_theta_p
  INTEGER :: n_rho_p, n_rho_mid_p
  REAL(wp), ALLOCATABLE :: rho_p(:)
  REAL(wp), ALLOCATABLE :: rho_mid_p(:)

  n_rho_p = ay_n_rho_p() + 1
  n_rho_mid_p = n_rho_p - 1
  ALLOCATE(rho_p(n_rho_p))
  ALLOCATE(rho_mid_p(n_rho_mid_p))
  
  CALL ay_sample_rho_p(rho_p)
  count_sample_pts_theta_p = 0

  do i=1, n_rho_mid_p
     rho_mid_p(i) = (rho_p(i+1) + rho_p(i))/2.0_wp
     count_sample_pts_theta_p = ay_n_theta_p(rho_mid_p(i)) + count_sample_pts_theta_p
  end do
  DEALLOCATE(rho_p)
  DEALLOCATE(rho_mid_p)
  ay_tot_num_obs_pts = count_sample_pts_theta_p
END FUNCTION ay_tot_num_obs_pts
!*****************************************************************************************************************************************

SUBROUTINE ay_obs_pts(obs_pts,surf_num)
  INTEGER, INTENT(IN) :: surf_num
  REAL(wp), INTENT(OUT) :: obs_pts(:,:)
  REAL(wp), ALLOCATABLE :: rho_p(:), rho_mid_p(:), theta_p(:)
  INTEGER :: i, j, n_rho_p, n_rho_mid_p, n_theta_p, n_theta_mid_p, row_index
  row_index = 1
  n_rho_p = ay_n_rho_p() + 1
  n_rho_mid_p = n_rho_p - 1
  ALLOCATE(rho_p(n_rho_p))
  ALLOCATE(rho_mid_p(n_rho_mid_p))

  CALL ay_sample_rho_p(rho_p)
  do i=1, n_rho_mid_p
     rho_mid_p(i) = (rho_p(i+1) + rho_p(i))/2.0_wp
     
     n_theta_p = ay_n_theta_p(rho_mid_p(i)) + 1
     n_theta_mid_p = n_theta_p - 1
     ALLOCATE(theta_p(n_theta_p))
     CALL ay_sample_theta_p(rho_mid_p(i),theta_p)
    
     do j=1, n_theta_mid_p
        obs_pts(row_index,1) = rho_mid_p(i)
        obs_pts(row_index,2) = (theta_p(j+1)+theta_p(j))/2.0_wp
        if( surf_num==1) then
           obs_pts(row_index,3) = S1(rho_mid_p(i), obs_pts(row_index,2) )
        end if
        if( surf_num==2) then
           obs_pts(row_index,3) = S2(rho_mid_p(i), obs_pts(row_index,2) )
        end if
        row_index = row_index + 1
     end do

     DEALLOCATE(theta_p)
  end do

  DEALLOCATE(rho_p)
  DEALLOCATE(rho_mid_p)
END SUBROUTINE ay_obs_pts
!*****************************************************************************************************************************************

  FUNCTION S1(xp_loc1,xp_loc2)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: S1
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 10/10/04
    ! LAST REVISION: 07/09/05
    ! DESCRIPTION:  Equation for Surface 1
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:                TYPE:                       COMMENT:
    ! xp_loc1             IN                    REAL                        rho in cylindrical coord. system
    ! xp_loc2             IN                    REAL                        theta in cylindrical coord. system
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp_loc1, xp_loc2
    REAL(wp)             :: S1
    
    S1= ay_c*Sqrt(1.0 - (xp_loc1/ay_R)**2)
    !Fancy Surface
    !& ay_c * (xp_loc1**4 * (sin(2.0*xp_loc2))**2 +.1) * Sqrt(1.0 - (xp_loc1/ay_R)**2)
  END FUNCTION S1
!*****************************************************************************************************************************************

  FUNCTION S2(xp_loc1,xp_loc2)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: S2
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 10/10/04
    ! LAST REVISION: 07/09/05
    ! DESCRIPTION:  Equation for Surface 2
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:                TYPE:                       COMMENT:
    ! xp_loc1             IN                    REAL                        rho in cylindrical coord. system
    ! xp_loc2             IN                    REAL                        theta in cylindrical coord. system
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp_loc1, xp_loc2
    REAL(wp)             :: S2
    
    S2= -ay_c*Sqrt(1.0 - (xp_loc1/ay_R)**2)
    !Fancy Surface         
    !&-1.0*ay_c * (xp_loc1**4 * (sin(2.0*xp_loc2))**2 + .1) * Sqrt(1.0 - (xp_loc1/ay_R)**2)
  END FUNCTION S2
!*****************************************************************************************************************************************

  FUNCTION n1x(x1_loc,x2_loc)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: n1x
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 10/10/04
    ! LAST REVISION: 07/09/05
    ! DESCRIPTION:  The x-component of the normal to Surface 1
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:                TYPE:                       COMMENT:
    ! x1_loc             IN                     REAL                        rho in cylindrical coord. system
    ! x2_loc             IN                     REAL                        theta in cylindrical coord. system
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    REAL(wp)             :: n1x

    n1x = (ay_c*x1_loc*Cos(x2_loc))/(ay_R*Sqrt(ay_R**2 - x1_loc** 2))
    !Fancy         
    !&(ay_c*x1_loc*Cos(x2_loc)*(0.1 + (-8.*ay_R**2*x1_loc**2 + 12.*x1_loc**4*Cos(x2_loc)**2)*Sin(x2_loc)**2 +& 
    !&8.*x1_loc**4*Sin(x2_loc)**4))/(ay_R*Sqrt(ay_R**2 - x1_loc**2))
  END FUNCTION n1x
!*****************************************************************************************************************************************

  FUNCTION n1y(x1_loc,x2_loc)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: n1y
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 10/10/04
    ! LAST REVISION: 07/09/05
    ! DESCRIPTION:  The y-component of the normal to Surface 1
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:                TYPE:                       COMMENT:
    ! x1_loc             IN                     REAL                        rho in cylindrical coord. system
    ! x2_loc             IN                     REAL                        theta in cylindrical coord. system
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    REAL(wp)             :: n1y

    n1y = (ay_c*x1_loc*Sin(x2_loc))/(ay_R*Sqrt(ay_R**2 - x1_loc**2))
    !Fancy         
    !&(ay_c*x1_loc*Sin(x2_loc)*(0.1 + 8.*x1_loc**4*Cos(x2_loc)**4 + Cos(x2_loc)**2*(-8.*ay_R**2*x1_loc**2 +& 
    !&12.*x1_loc**4*Sin(x2_loc)**2)))/(ay_R*Sqrt(ay_R**2 - x1_loc**2))
  END FUNCTION n1y
!*****************************************************************************************************************************************

  FUNCTION n1z(x1_loc,x2_loc)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: n1z
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 10/10/04
    ! LAST REVISION: 
    ! DESCRIPTION:  The z-component of the normal to Surface 1
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:                TYPE:                       COMMENT:
    ! x1_loc             IN                     REAL                        rho in cylindrical coord. system
    ! x2_loc             IN                     REAL                        theta in cylindrical coord. system
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    REAL(wp)             :: n1z

    n1z = 1.0_wp
  END FUNCTION n1z
!*****************************************************************************************************************************************

  FUNCTION mag_sqred_n1(x1_loc,x2_loc)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: mag_sqred_n1
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 07/09/05
    ! LAST REVISION: 
    ! DESCRIPTION: Computes magnituted SQUARED of normal n1
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:               TYPE:                       COMMENT:
    ! x1_loc             IN                    REAL                        rho in cylindrical coord. system
    ! x2_loc             IN                    REAL                        theta in cylindrical coord. system
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    REAL(wp)             :: mag_sqred_n1

    mag_sqred_n1 = (n1x(x1_loc,x2_loc))**2 + (n1y(x1_loc,x2_loc))**2 + (n1z(x1_loc,x2_loc))**2
  END FUNCTION mag_sqred_n1


  FUNCTION n2x(x1_loc,x2_loc)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: n2x
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 10/10/04
    ! LAST REVISION: 07/09/05
    ! DESCRIPTION:  The x-component of the normal to Surface 2
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:                TYPE:                       COMMENT:
    ! x1_loc             IN                     REAL                        rho in cylindrical coord. system
    ! x2_loc             IN                     REAL                        theta in cylindrical coord. system
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    REAL(wp)             :: n2x

    n2x = (ay_c*x1_loc*Cos(x2_loc))/(ay_R*Sqrt(ay_R**2 - x1_loc**2))
    !Fancy         
    !&(ay_c*x1_loc*Cos(x2_loc)*(0.1 + (-8.*ay_R**2*x1_loc**2 + 12.*x1_loc**4*Cos(x2_loc)**2)*Sin(x2_loc)**2 +& 
    !&8.*x1_loc**4*Sin(x2_loc)**4))/(ay_R*Sqrt(ay_R**2 - x1_loc**2))
  END FUNCTION n2x
!*****************************************************************************************************************************************

  FUNCTION n2y(x1_loc,x2_loc)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: n2y
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 10/10/04
    ! LAST REVISION: 07/09/05
    ! DESCRIPTION:  The y-component of the normal to Surface 2
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:                TYPE:                       COMMENT:
    ! x1_loc             IN                     REAL                        rho in cylindrical coord. system
    ! x2_loc             IN                     REAL                        theta in cylindrical coord. system
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    REAL(wp)             :: n2y

    n2y = (ay_c*x1_loc*Sin(x2_loc))/(ay_R*Sqrt(ay_R**2 - x1_loc**2))
    !Fancy    
    !&(ay_c*x1_loc*Sin(x2_loc)*(0.1 + 8.*x1_loc**4*Cos(x2_loc)**4 + Cos(x2_loc)**2*(-8.*ay_R**2*x1_loc**2 +& 
    !&12.*x1_loc**4*Sin(x2_loc)**2)))/(ay_R*Sqrt(ay_R**2 - x1_loc**2))
  END FUNCTION n2y
!*****************************************************************************************************************************************

  FUNCTION n2z(x1_loc,x2_loc)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: n2z
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 10/10/04
    ! LAST REVISION: 
    ! DESCRIPTION:  The z-component of the normal to Surface 2
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:                TYPE:                       COMMENT:
    ! x1_loc             IN                     REAL                        rho in cylindrical coord. system
    ! x2_loc             IN                     REAL                        theta in cylindrical coord. system
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    REAL(wp)             :: n2z

    n2z = -1.0_wp
  END FUNCTION n2z
!*****************************************************************************************************************************************

  FUNCTION mag_sqred_n2(x1_loc,x2_loc)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: mag_sqred_n2
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 07/09/05
    ! LAST REVISION: 
    ! DESCRIPTION: Computes magnituted SQUARED of normal n1
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:               TYPE:                       COMMENT:
    ! x1_loc             IN                    REAL                        rho in cylindrical coord. system
    ! x2_loc             IN                    REAL                        theta in cylindrical coord. system
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    REAL(wp)             :: mag_sqred_n2

    mag_sqred_n2 = (n2x(x1_loc,x2_loc))**2 + (n2y(x1_loc,x2_loc))**2 + (n2z(x1_loc,x2_loc))**2
  END FUNCTION mag_sqred_n2
!*****************************************************************************************************************************************


  FUNCTION c11_1(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          :: c11_1
    c11_1 = 1.0 + (1.0/ay_epsilon -1.0)*(n1x(x1_loc,x2_loc))**2/mag_sqred_n1(x1_loc,x2_loc)
  END FUNCTION c11_1
!*****************************************************************************************************************************************

  FUNCTION c12_1(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          :: c12_1
    c12_1 = (1.0/ay_epsilon -1.0)*( n1x(x1_loc,x2_loc)*n1y(x1_loc,x2_loc) )/mag_sqred_n1(x1_loc,x2_loc)
  END FUNCTION c12_1
!*****************************************************************************************************************************************

  FUNCTION c13_1(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          :: c13_1
    c13_1 = (1.0/ay_epsilon -1.0)*( n1x(x1_loc,x2_loc)*n1z(x1_loc,x2_loc) )/mag_sqred_n1(x1_loc,x2_loc)
  END FUNCTION c13_1
!*****************************************************************************************************************************************

  FUNCTION c22_1(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          :: c22_1
    c22_1 = 1.0 + (1.0/ay_epsilon -1.0)*(n1y(x1_loc,x2_loc))**2/mag_sqred_n1(x1_loc,x2_loc)
  END FUNCTION C22_1
!*****************************************************************************************************************************************

  FUNCTION c23_1(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          :: c23_1
    c23_1 = (1.0/ay_epsilon -1.0)*( n1y(x1_loc,x2_loc)*n1z(x1_loc,x2_loc) )/mag_sqred_n1(x1_loc,x2_loc)
  END FUNCTION c23_1
!*****************************************************************************************************************************************

  FUNCTION c33_1(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          ::c33_1
    c33_1 = 1.0 + (1.0/ay_epsilon -1.0)*(n1z(x1_loc,x2_loc))**2/mag_sqred_n1(x1_loc,x2_loc)
  END FUNCTION c33_1
!*****************************************************************************************************************************************

  FUNCTION c11_2(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          :: c11_2
    c11_2 = 1.0 + (1.0/ay_epsilon -1.0)*(n2x(x1_loc,x2_loc))**2/mag_sqred_n2(x1_loc,x2_loc)
  END FUNCTION c11_2
!*****************************************************************************************************************************************

  FUNCTION c12_2(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          :: c12_2
    c12_2 = (1.0/ay_epsilon -1.0)*( n2x(x1_loc,x2_loc)*n2y(x1_loc,x2_loc) )/mag_sqred_n2(x1_loc,x2_loc)
  END FUNCTION c12_2
!*****************************************************************************************************************************************

  FUNCTION c13_2(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          :: c13_2
    c13_2 = (1.0/ay_epsilon -1.0)*( n2x(x1_loc,x2_loc)*n2z(x1_loc,x2_loc) )/mag_sqred_n2(x1_loc,x2_loc)
  END FUNCTION c13_2
!*****************************************************************************************************************************************

  FUNCTION c22_2(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          :: c22_2
    c22_2 = 1.0 + (1.0/ay_epsilon -1.0)*(n2y(x1_loc,x2_loc))**2/mag_sqred_n2(x1_loc,x2_loc)
  END FUNCTION C22_2
!*****************************************************************************************************************************************

  FUNCTION c23_2(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          :: c23_2
    c23_2 =  (1.0/ay_epsilon -1.0)*( n2y(x1_loc,x2_loc)*n2z(x1_loc,x2_loc) )/mag_sqred_n2(x1_loc,x2_loc)
  END FUNCTION c23_2
!*****************************************************************************************************************************************

  FUNCTION c33_2(x1_loc,x2_loc)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: x1_loc, x2_loc
    COMPLEX(wp)          ::c33_2
    c33_2 = 1.0 + (1.0/ay_epsilon -1.0)*(n2z(x1_loc,x2_loc))**2/mag_sqred_n2(x1_loc,x2_loc)
  END FUNCTION c33_2
!*****************************************************************************************************************************************


END MODULE ay_integration
