!-----------------------------------------------------------------------------------------------------------------------------------------
! TITLE: ay_kernels.f95
! AUTHOR: Alex Yuffa
! DATE WRITTEN: 08/08/04
! LAST REVISION: 07/09/05
! DESCRIPTION:  Contains kernels for integral equations.
!-----------------------------------------------------------------------------------------------------------------------------------------

!-----------------------------------------------------------------------------------------------------------------------------------------
! NON-LOCAL VARIABLES USED:
! MODULE NAME:            VARIABLE NAME:      TYPE:                   COMMENT:
! ay_kind                 wp                  INTEGER, PARAMETER      Working precision
! ay_constants            ay_pi               REAL, PARAMETER         Value of Pi
! ay_constants            ay_k1               REAL, PARAMETER         Wavenumber in Region 1
! ay_constants            ay_k2               REAL, PARAMETER         Wavenumber in Region 2
! ay_constants            ay_I                COMPLEX, PARAMETER      Sqrt(-1)
!-----------------------------------------------------------------------------------------------------------------------------------------

!-----------------------------------------------------------------------------------------------------------------------------------------
! LOCAL VARIABLES USED:
! VARIABLE NAME:                 TYPE:               COMMENT:
! x                              REAL, ARRAY         Observation Point.  x(1) = rho, x(2) = theta, x(3) = z
!-----------------------------------------------------------------------------------------------------------------------------------------

MODULE ay_kernels
  USE ay_kind, ONLY : wp
  USE ay_constants, ONLY : ay_pi, ay_k1, ay_k2, ay_I, ay_mu, ay_epsilon
  USE ay_integration, ONLY : S1, S2, n1x, n1y, n1z, n2x, n2y, n2z 
  IMPLICIT NONE
  
  REAL(wp), PRIVATE, SAVE :: x(3) !Observation point
  
CONTAINS

  SUBROUTINE ay_write_obs_pt(x_loc)
    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: ay_write_obs_pt
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 08/08/04
    ! LAST REVISION: 11/21/04 
    ! DESCRIPTION: Writes observation point so integral equations kernels can use it.
    ! INPUT/OUTPUT VARIABLES:
    ! NAME:              INTENT:            TYPE:               COMMENT:
    ! x_loc              IN                 REAL, ARRAY         Writes x_loc to x (Observation Point in this module, with PRIVATE and
    !                                                           SAVE attributes.                                                          
    !-------------------------------------------------------------------------------------------------------------------------------------
    IMPLICIT NONE

    REAL(wp), INTENT(IN) :: x_loc(:)
    x(:) = x_loc(:)
  END SUBROUTINE ay_write_obs_pt
!*****************************************************************************************************************************************

    !-------------------------------------------------------------------------------------------------------------------------------------
    ! TITLE: ay_UL1x1x
    ! AUTHOR: Alex Yuffa
    ! DATE WRITTEN: 10/10/04
    ! LAST REVISION: 12/31/04
    ! DESCRIPTION:  Integral Equation Kernel, Normal Derivative of the Green's Function on Surface 1, Observation point on Surface 1
    ! INPUT/OUTPUT VARIABLES:
    ! 
    !-------------------------------------------------------------------------------------------------------------------------------------

!-------------------------------------------------------------UL BLOCK -------------------------------------------------------------------

  FUNCTION ay_UL1x1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_UL1x1x(m)
    COMPLEX(wp) :: UL1x1x

    UL1x1x = &
&(Exp(ay_I*ay_k1*Sqrt(x(1)**2 + (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*xp(1)*(n1z(xp(1),xp(2))*(S1(xp(1),xp(2)) - x(3)) + n1x(xp(1),xp(2))*(-(Cos(x(2))*x(1)) + Cos(xp&
&(2))*xp(1)) + n1y(xp(1),xp(2))*(-(Sin(x(2))*x(1)) + Sin(xp(2))*xp(1)))*(1 - ay_I*ay_k1*Sqrt(x(1)**2 &
&+ (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(4.*ay_pi*(x(1)**2 + (S&
&1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_UL1x1x = (/ REAL(UL1x1x,wp), AIMAG(UL1x1x) /)
  END FUNCTION AY_UL1X1X
!*****************************************************************************************************************************************

  FUNCTION ay_UL1x2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_UL1x2x(m)
    COMPLEX(wp) :: UL1x2x

    UL1x2x = &
&(Exp(ay_I*ay_k1*Sqrt(x(1)**2 + (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*xp(1)*(n2z(xp(1),xp(2))*(S2(xp(1),xp(2)) - x(3)) + n2x(xp(1),xp(2))*(-(Cos(x(2))*x(1)) + Cos(xp&
&(2))*xp(1)) + n2y(xp(1),xp(2))*(-(Sin(x(2))*x(1)) + Sin(xp(2))*xp(1)))*(1 - ay_I*ay_k1*Sqrt(x(1)**2 &
&+ (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(4.*ay_pi*(x(1)**2 + (S&
&2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)
    
    ay_UL1x2x = (/ REAL(UL1x2x,wp), AIMAG(UL1x2x) /)    
  END FUNCTION AY_UL1X2X
!*****************************************************************************************************************************************

  FUNCTION ay_UL2x1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_UL2x1x(m)
    COMPLEX(wp) :: UL2x1x

    UL2x1x = &
&(Exp(ay_I*ay_k1*Sqrt(x(1)**2 + (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*xp(1)*(n1z(xp(1),xp(2))*(S1(xp(1),xp(2)) - x(3)) + n1x(xp(1),xp(2))*(-(Cos(x(2))*x(1)) + Cos(xp&
&(2))*xp(1)) + n1y(xp(1),xp(2))*(-(Sin(x(2))*x(1)) + Sin(xp(2))*xp(1)))*(1 - ay_I*ay_k1*Sqrt(x(1)**2 &
&+ (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(4.*ay_pi*(x(1)**2 + (S&
&1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_UL2x1x = (/ REAL(UL2x1x,wp), AIMAG(UL2x1x) /)    
  END FUNCTION AY_UL2X1X
!*****************************************************************************************************************************************

  FUNCTION ay_UL2x2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_UL2x2x(m)
    COMPLEX(wp) :: UL2x2x

    UL2x2x = &
&(Exp(ay_I*ay_k1*Sqrt(x(1)**2 + (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*xp(1)*(n2z(xp(1),xp(2))*(S2(xp(1),xp(2)) - x(3)) + n2x(xp(1),xp(2))*(-(Cos(x(2))*x(1)) + Cos(xp&
&(2))*xp(1)) + n2y(xp(1),xp(2))*(-(Sin(x(2))*x(1)) + Sin(xp(2))*xp(1)))*(1 - ay_I*ay_k1*Sqrt(x(1)**2 &
&+ (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(4.*ay_pi*(x(1)**2 + (S&
&2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_UL2x2x = (/ REAL(UL2x2x,wp), AIMAG(UL2x2x) /)    
  END FUNCTION AY_UL2X2X
!*****************************************************************************************************************************************

!-------------------------------------------------------------UR BLOCK -------------------------------------------------------------------

  FUNCTION ay_UR1x1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_UR1x1x(m)
    COMPLEX(wp) :: UR1x1x

    UR1x1x = &
&(Exp(ay_I*ay_k1*Sqrt(x(1)**2 + (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*xp(1))/(4.*ay_pi*Sqrt(x(1)**2 + (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2))
    
    ay_UR1x1x = (/ REAL(UR1x1x,wp), AIMAG(UR1x1x) /)    
  END FUNCTION AY_UR1X1X
!*****************************************************************************************************************************************

  FUNCTION ay_UR1x2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_UR1x2x(m)
    COMPLEX(wp) :: UR1x2x

    UR1x2x = &
&(Exp(ay_I*ay_k1*Sqrt(x(1)**2 + (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*xp(1))/(4.*ay_pi*Sqrt(x(1)**2 + (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2))

         ay_UR1x2x = (/ REAL(UR1x2x,wp), AIMAG(UR1x2x) /)    
  END FUNCTION AY_UR1X2X
!*****************************************************************************************************************************************

  FUNCTION ay_UR2x1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_UR2x1x(m)
    COMPLEX(wp) :: UR2x1x

    UR2x1x = &
&(Exp(ay_I*ay_k1*Sqrt(x(1)**2 + (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*xp(1))/(4.*ay_pi*Sqrt(x(1)**2 + (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2))

    ay_UR2x1x = (/ REAL(UR2x1x,wp), AIMAG(UR2x1x) /)    
  END FUNCTION AY_UR2X1X
!*****************************************************************************************************************************************

  FUNCTION ay_UR2x2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_UR2x2x(m)
    COMPLEX(wp) :: UR2x2x

    UR2x2x = &
&(Exp(ay_I*ay_k1*Sqrt(x(1)**2 + (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*xp(1))/(4.*ay_pi*Sqrt(x(1)**2 + (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2))
 
    ay_UR2x2x = (/ REAL(UR2x2x,wp), AIMAG(UR2x2x) /)    
  END FUNCTION AY_UR2X2X
!*****************************************************************************************************************************************

!-------------------------------------------------------------LR BLOCK -------------------------------------------------------------------

  FUNCTION ay_LR1x1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LR1x1x(m)
    COMPLEX(wp) :: LR1x1x

    LR1x1x = &
&(Exp(ay_I*ay_k2*Sqrt(x(1)**2 + (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*ay_mu*xp(1))/(4.*ay_pi*Sqrt(x(1)**2 + (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp&
&(1) + xp(1)**2))

    ay_LR1x1x = (/ REAL(LR1x1x,wp), AIMAG(LR1x1x) /)    
  END FUNCTION AY_LR1X1X
!*****************************************************************************************************************************************

  FUNCTION ay_LR1x2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LR1x2x(m)
    COMPLEX(wp) :: LR1x2x

    LR1x2x = &
&(Exp(ay_I*ay_k2*Sqrt(x(1)**2 + (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*ay_mu*xp(1))/(4.*ay_pi*Sqrt(x(1)**2 + (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp&
&(1) + xp(1)**2))

    ay_LR1x2x = (/ REAL(LR1x2x,wp), AIMAG(LR1x2x) /)    
  END FUNCTION AY_LR1X2X
!*****************************************************************************************************************************************

  FUNCTION ay_LR2x1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LR2x1x(m)
    COMPLEX(wp) :: LR2x1x

    LR2x1x = &
&(Exp(ay_I*ay_k2*Sqrt(x(1)**2 + (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*ay_mu*xp(1))/(4.*ay_pi*Sqrt(x(1)**2 + (S1(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp&
&(1) + xp(1)**2))

    ay_LR2x1x = (/ REAL(LR2x1x,wp), AIMAG(LR2x1x) /)    
  END FUNCTION AY_LR2X1X
!*****************************************************************************************************************************************

  FUNCTION ay_LR2x2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LR2x2x(m)
    COMPLEX(wp) :: LR2x2x

    LR2x2x = &
&(Exp(ay_I*ay_k2*Sqrt(x(1)**2 + (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)*&
&*2))*ay_mu*xp(1))/(4.*ay_pi*Sqrt(x(1)**2 + (S2(xp(1),xp(2)) - x(3))**2 - 2*Cos(x(2) - xp(2))*x(1)*xp&
&(1) + xp(1)**2))

    ay_LR2x2x = (/ REAL(LR2x2x,wp), AIMAG(LR2x2x) /)    
  END FUNCTION AY_LR2X2X
!*****************************************************************************************************************************************

!-------------------------------------------------------------LL BLOCK -------------------------------------------------------------------

  FUNCTION ay_LL1x1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1x1x(m)
    COMPLEX(wp) :: LL1x1x

    LL1x1x = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*n1z(xp(1),xp(2))*S1(xp(1),xp(2)) - n1x(xp(1),xp(2))&
&*Cos(x(2))*x(1) - ay_epsilon*n1y(xp(1),xp(2))*Sin(x(2))*x(1) - ay_epsilon*n1z(xp(1),xp(2))*x(3) + n1&
&x(xp(1),xp(2))*Cos(xp(2))*xp(1) + ay_epsilon*n1y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1x1x = (/ REAL(LL1x1x,wp), AIMAG(LL1x1x) /)    
  END FUNCTION AY_LL1X1X
!*****************************************************************************************************************************************

  FUNCTION ay_LL1x1y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1x1y(m)
    COMPLEX(wp) :: LL1x1y

    LL1x1y = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n1y(xp(1),xp(2))*(Cos(x(2))*x(1) - Cos&
&(xp(2))*xp(1)) - ay_epsilon*(-1 + ay_mu)*n1x(xp(1),xp(2))*(Sin(x(2))*x(1) - Sin(xp(2))*xp(1)))*(1 - &
&ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2)&
&)*x(1)*xp(1) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(&
&3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1x1y = (/ REAL(LL1x1y,wp), AIMAG(LL1x1y) /)    
  END FUNCTION AY_LL1X1Y
!*****************************************************************************************************************************************

  FUNCTION ay_LL1x1z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1x1z(m)
    COMPLEX(wp) :: LL1x1z

    LL1x1z = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n1x(xp(1),xp(2))*(S1(xp(1),xp(2)) - x(&
&3)) + (-1 + ay_epsilon*ay_mu)*n1z(xp(1),xp(2))*(Cos(x(2))*x(1) - Cos(xp(2))*xp(1)))*(1 - ay_I*ay_k2*&
&Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1&
&) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**&
&2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1x1z = (/ REAL(LL1x1z,wp), AIMAG(LL1x1z) /)    
  END FUNCTION AY_LL1X1Z
!*****************************************************************************************************************************************

  FUNCTION ay_LL1x2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1x2x(m)
    COMPLEX(wp) :: LL1x2x

    LL1x2x = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*n2z(xp(1),xp(2))*S2(xp(1),xp(2)) - n2x(xp(1),xp(2))&
&*Cos(x(2))*x(1) - ay_epsilon*n2y(xp(1),xp(2))*Sin(x(2))*x(1) - ay_epsilon*n2z(xp(1),xp(2))*x(3) + n2&
&x(xp(1),xp(2))*Cos(xp(2))*xp(1) + ay_epsilon*n2y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1x2x = (/ REAL(LL1x2x,wp), AIMAG(LL1x2x) /)    
  END FUNCTION AY_LL1X2X
!*****************************************************************************************************************************************

  FUNCTION ay_LL1x2y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1x2y(m)
    COMPLEX(wp) :: LL1x2y

    LL1x2y = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n2y(xp(1),xp(2))*(Cos(x(2))*x(1) - Cos&
&(xp(2))*xp(1)) - ay_epsilon*(-1 + ay_mu)*n2x(xp(1),xp(2))*(Sin(x(2))*x(1) - Sin(xp(2))*xp(1)))*(1 - &
&ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2)&
&)*x(1)*xp(1) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(&
&3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1x2y = (/ REAL(LL1x2y,wp), AIMAG(LL1x2y) /)    
  END FUNCTION AY_LL1X2Y
!*****************************************************************************************************************************************

  FUNCTION ay_LL1x2z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1x2z(m)
    COMPLEX(wp) :: LL1x2z

    LL1x2z = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n2x(xp(1),xp(2))*(S2(xp(1),xp(2)) - x(&
&3)) + (-1 + ay_epsilon*ay_mu)*n2z(xp(1),xp(2))*(Cos(x(2))*x(1) - Cos(xp(2))*xp(1)))*(1 - ay_I*ay_k2*&
&Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1&
&) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**&
&2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)
         
    ay_LL1x2z = (/ REAL(LL1x2z,wp), AIMAG(LL1x2z) /)    
  END FUNCTION AY_LL1X2Z
!*****************************************************************************************************************************************

  FUNCTION ay_LL1y1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1y1x(m)
    COMPLEX(wp) :: LL1y1x

    LL1y1x = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n1y(xp(1),xp(2))*Cos(x(2))*x(&
&1) - (-1 + ay_epsilon*ay_mu)*n1x(xp(1),xp(2))*Sin(x(2))*x(1) + (-(ay_epsilon*(-1 + ay_mu)*n1y(xp(1),&
&xp(2))*Cos(xp(2))) + (-1 + ay_epsilon*ay_mu)*n1x(xp(1),xp(2))*Sin(xp(2)))*xp(1))*(ay_I + ay_k2*Sqrt(&
&S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + x&
&p(1)**2)))/(ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)
    
    ay_LL1y1x = (/ REAL(LL1y1x,wp), AIMAG(LL1y1x) /)    
  END FUNCTION AY_LL1Y1X
!*****************************************************************************************************************************************

  FUNCTION ay_LL1y1y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1y1y(m)
    COMPLEX(wp) :: LL1y1y

    LL1y1y = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*n1z(xp(1),xp(2))*S1(xp(1),xp(2)) - ay_epsilon*n1x(x&
&p(1),xp(2))*Cos(x(2))*x(1) - n1y(xp(1),xp(2))*Sin(x(2))*x(1) - ay_epsilon*n1z(xp(1),xp(2))*x(3) + ay&
&_epsilon*n1x(xp(1),xp(2))*Cos(xp(2))*xp(1) + n1y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1y1y = (/ REAL(LL1y1y,wp), AIMAG(LL1y1y) /)    
  END FUNCTION Ay_LL1y1y
!*****************************************************************************************************************************************

  FUNCTION ay_LL1y1z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1y1z(m)
    COMPLEX(wp) :: LL1y1z

    LL1y1z = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n1y(xp(1),xp(2))*(S1(xp(1),xp(2)) - x(&
&3)) + (-1 + ay_epsilon*ay_mu)*n1z(xp(1),xp(2))*(Sin(x(2))*x(1) - Sin(xp(2))*xp(1)))*(1 - ay_I*ay_k2*&
&Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1&
&) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**&
&2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1y1z = (/ REAL(LL1y1z,wp), AIMAG(LL1y1z) /)    
  END FUNCTION AY_LL1Y1Z
!*****************************************************************************************************************************************

  FUNCTION ay_LL1y2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1y2x(m)
    COMPLEX(wp) :: LL1y2x

    LL1y2x = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n2y(xp(1),xp(2))*Cos(x(2))*x(&
&1) - (-1 + ay_epsilon*ay_mu)*n2x(xp(1),xp(2))*Sin(x(2))*x(1) + (-(ay_epsilon*(-1 + ay_mu)*n2y(xp(1),&
&xp(2))*Cos(xp(2))) + (-1 + ay_epsilon*ay_mu)*n2x(xp(1),xp(2))*Sin(xp(2)))*xp(1))*(ay_I + ay_k2*Sqrt(&
&S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + x&
&p(1)**2)))/(ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1y2x = (/ REAL(LL1y2x,wp), AIMAG(LL1y2x) /)    
  END FUNCTION AY_LL1Y2X
!*****************************************************************************************************************************************

  FUNCTION ay_LL1y2y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1y2y(m)
    COMPLEX(wp) :: LL1y2y

    LL1y2y = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*n2z(xp(1),xp(2))*S2(xp(1),xp(2)) - ay_epsilon*n2x(x&
&p(1),xp(2))*Cos(x(2))*x(1) - n2y(xp(1),xp(2))*Sin(x(2))*x(1) - ay_epsilon*n2z(xp(1),xp(2))*x(3) + ay&
&_epsilon*n2x(xp(1),xp(2))*Cos(xp(2))*xp(1) + n2y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1y2y = (/ REAL(LL1y2y,wp), AIMAG(LL1y2y) /)    
  END FUNCTION AY_LL1Y2Y
!*****************************************************************************************************************************************

  FUNCTION ay_LL1y2z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1y2z(m)
    COMPLEX(wp) :: LL1y2z

    LL1y2z = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n2y(xp(1),xp(2))*(S2(xp(1),xp(2)) - x(&
&3)) + (-1 + ay_epsilon*ay_mu)*n2z(xp(1),xp(2))*(Sin(x(2))*x(1) - Sin(xp(2))*xp(1)))*(1 - ay_I*ay_k2*&
&Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1&
&) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**&
&2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1y2z = (/ REAL(LL1y2z,wp), AIMAG(LL1y2z) /)    
  END FUNCTION AY_LL1Y2Z
!*****************************************************************************************************************************************

  FUNCTION ay_LL1z1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1z1x(m)
    COMPLEX(wp) :: LL1z1x

    LL1z1x = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n1x(xp(1),xp(2))*S1(xp(1),xp(&
&2)) + ay_epsilon*(-1 + ay_mu)*n1z(xp(1),xp(2))*Cos(x(2))*x(1) + n1x(xp(1),xp(2))*x(3) - ay_epsilon*a&
&y_mu*n1x(xp(1),xp(2))*x(3) + ay_epsilon*n1z(xp(1),xp(2))*Cos(xp(2))*xp(1) - ay_epsilon*ay_mu*n1z(xp(&
&1),xp(2))*Cos(xp(2))*xp(1))*(ay_I + ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3)&
& + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(&
&1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1z1x = (/ REAL(LL1z1x,wp), AIMAG(LL1z1x) /)    
  END FUNCTION AY_LL1Z1X
!*****************************************************************************************************************************************

  FUNCTION ay_LL1z1y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1z1y(m)
    COMPLEX(wp) :: LL1z1y

    LL1z1y = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n1y(xp(1),xp(2))*S1(xp(1),xp(&
&2)) + ay_epsilon*(-1 + ay_mu)*n1z(xp(1),xp(2))*Sin(x(2))*x(1) + n1y(xp(1),xp(2))*x(3) - ay_epsilon*a&
&y_mu*n1y(xp(1),xp(2))*x(3) + ay_epsilon*n1z(xp(1),xp(2))*Sin(xp(2))*xp(1) - ay_epsilon*ay_mu*n1z(xp(&
&1),xp(2))*Sin(xp(2))*xp(1))*(ay_I + ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3)&
& + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(&
&1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1z1y = (/ REAL(LL1z1y,wp), AIMAG(LL1z1y) /)    
  END FUNCTION AY_LL1Z1Y
!*****************************************************************************************************************************************

  FUNCTION ay_LL1z1z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1z1z(m)
    COMPLEX(wp) :: LL1z1z

    LL1z1z = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(n1z(xp(1),xp(2))*S1(xp(1),xp(2)) - ay_epsilon*n1x(xp(1),xp(2))&
&*Cos(x(2))*x(1) - ay_epsilon*n1y(xp(1),xp(2))*Sin(x(2))*x(1) - n1z(xp(1),xp(2))*x(3) + ay_epsilon*n1&
&x(xp(1),xp(2))*Cos(xp(2))*xp(1) + ay_epsilon*n1y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1z1z = (/ REAL(LL1z1z,wp), AIMAG(LL1z1z) /)    
  END FUNCTION AY_LL1Z1Z
!*****************************************************************************************************************************************

  FUNCTION ay_LL1z2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1z2x(m)
    COMPLEX(wp) :: LL1z2x

    LL1z2x = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n2x(xp(1),xp(2))*S2(xp(1),xp(&
&2)) + ay_epsilon*(-1 + ay_mu)*n2z(xp(1),xp(2))*Cos(x(2))*x(1) + n2x(xp(1),xp(2))*x(3) - ay_epsilon*a&
&y_mu*n2x(xp(1),xp(2))*x(3) + ay_epsilon*n2z(xp(1),xp(2))*Cos(xp(2))*xp(1) - ay_epsilon*ay_mu*n2z(xp(&
&1),xp(2))*Cos(xp(2))*xp(1))*(ay_I + ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3)&
& + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(&
&1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1z2x = (/ REAL(LL1z2x,wp), AIMAG(LL1z2x) /)    
  END FUNCTION AY_LL1Z2X
!*****************************************************************************************************************************************

  FUNCTION ay_LL1z2y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1z2y(m)
    COMPLEX(wp) :: LL1z2y

    LL1z2y = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n2y(xp(1),xp(2))*S2(xp(1),xp(&
&2)) + ay_epsilon*(-1 + ay_mu)*n2z(xp(1),xp(2))*Sin(x(2))*x(1) + n2y(xp(1),xp(2))*x(3) - ay_epsilon*a&
&y_mu*n2y(xp(1),xp(2))*x(3) + ay_epsilon*n2z(xp(1),xp(2))*Sin(xp(2))*xp(1) - ay_epsilon*ay_mu*n2z(xp(&
&1),xp(2))*Sin(xp(2))*xp(1))*(ay_I + ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3)&
& + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(&
&1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1z2y = (/ REAL(LL1z2y,wp), AIMAG(LL1z2y) /)    
  END FUNCTION AY_LL1Z2Y
!*****************************************************************************************************************************************

  FUNCTION ay_LL1z2z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL1z2z(m)
    COMPLEX(wp) :: LL1z2z

    LL1z2z = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(n2z(xp(1),xp(2))*S2(xp(1),xp(2)) - ay_epsilon*n2x(xp(1),xp(2))&
&*Cos(x(2))*x(1) - ay_epsilon*n2y(xp(1),xp(2))*Sin(x(2))*x(1) - n2z(xp(1),xp(2))*x(3) + ay_epsilon*n2&
&x(xp(1),xp(2))*Cos(xp(2))*xp(1) + ay_epsilon*n2y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL1z2z = (/ REAL(LL1z2z,wp), AIMAG(LL1z2z) /)    
  END FUNCTION AY_LL1Z2Z
!*****************************************************************************************************************************************

  FUNCTION ay_LL2x1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2x1x(m)
    COMPLEX(wp) :: LL2x1x

    LL2x1x = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*n1z(xp(1),xp(2))*S1(xp(1),xp(2)) - n1x(xp(1),xp(2))&
&*Cos(x(2))*x(1) - ay_epsilon*n1y(xp(1),xp(2))*Sin(x(2))*x(1) - ay_epsilon*n1z(xp(1),xp(2))*x(3) + n1&
&x(xp(1),xp(2))*Cos(xp(2))*xp(1) + ay_epsilon*n1y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2x1x = (/ REAL(LL2x1x,wp), AIMAG(LL2x1x) /)    
  END FUNCTION AY_LL2X1X
!*****************************************************************************************************************************************

  FUNCTION ay_LL2x1y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2x1y(m)
    COMPLEX(wp) :: LL2x1y

    LL2x1y = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n1y(xp(1),xp(2))*(Cos(x(2))*x(1) - Cos&
&(xp(2))*xp(1)) - ay_epsilon*(-1 + ay_mu)*n1x(xp(1),xp(2))*(Sin(x(2))*x(1) - Sin(xp(2))*xp(1)))*(1 - &
&ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2)&
&)*x(1)*xp(1) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(&
&3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2x1y = (/ REAL(LL2x1y,wp), AIMAG(LL2x1y) /)    
  END FUNCTION AY_LL2X1Y
!*****************************************************************************************************************************************

  FUNCTION ay_LL2x1z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2x1z(m)
    COMPLEX(wp) :: LL2x1z

    LL2x1z = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n1x(xp(1),xp(2))*(S1(xp(1),xp(2)) - x(&
&3)) + (-1 + ay_epsilon*ay_mu)*n1z(xp(1),xp(2))*(Cos(x(2))*x(1) - Cos(xp(2))*xp(1)))*(1 - ay_I*ay_k2*&
&Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1&
&) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**&
&2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2x1z = (/ REAL(LL2x1z,wp), AIMAG(LL2x1z) /)    
  END FUNCTION AY_LL2x1z
!*****************************************************************************************************************************************

  FUNCTION ay_LL2x2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2x2x(m)
    COMPLEX(wp) :: LL2x2x

    LL2x2x = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*n2z(xp(1),xp(2))*S2(xp(1),xp(2)) - n2x(xp(1),xp(2))&
&*Cos(x(2))*x(1) - ay_epsilon*n2y(xp(1),xp(2))*Sin(x(2))*x(1) - ay_epsilon*n2z(xp(1),xp(2))*x(3) + n2&
&x(xp(1),xp(2))*Cos(xp(2))*xp(1) + ay_epsilon*n2y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2x2x = (/ REAL(LL2x2x,wp), AIMAG(LL2x2x) /)    
  END FUNCTION Ay_LL2x2x
!*****************************************************************************************************************************************

  FUNCTION ay_LL2x2y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2x2y(m)
    COMPLEX(wp) :: LL2x2y

    LL2x2y = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n2y(xp(1),xp(2))*(Cos(x(2))*x(1) - Cos&
&(xp(2))*xp(1)) - ay_epsilon*(-1 + ay_mu)*n2x(xp(1),xp(2))*(Sin(x(2))*x(1) - Sin(xp(2))*xp(1)))*(1 - &
&ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2)&
&)*x(1)*xp(1) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(&
&3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2x2y = (/ REAL(LL2x2y,wp), AIMAG(LL2x2y) /)    
  END FUNCTION AY_LL2X2Y
!*****************************************************************************************************************************************

  FUNCTION ay_LL2x2z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2x2z(m)
    COMPLEX(wp) :: LL2x2z

    LL2x2z = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n2x(xp(1),xp(2))*(S2(xp(1),xp(2)) - x(&
&3)) + (-1 + ay_epsilon*ay_mu)*n2z(xp(1),xp(2))*(Cos(x(2))*x(1) - Cos(xp(2))*xp(1)))*(1 - ay_I*ay_k2*&
&Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1&
&) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**&
&2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2x2z = (/ REAL(LL2x2z,wp), AIMAG(LL2x2z) /)    
  END FUNCTION AY_LL2X2Z
!*****************************************************************************************************************************************

  FUNCTION ay_LL2y1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2y1x(m)
    COMPLEX(wp) :: LL2y1x

    LL2y1x = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n1y(xp(1),xp(2))*Cos(x(2))*x(&
&1) - (-1 + ay_epsilon*ay_mu)*n1x(xp(1),xp(2))*Sin(x(2))*x(1) + (-(ay_epsilon*(-1 + ay_mu)*n1y(xp(1),&
&xp(2))*Cos(xp(2))) + (-1 + ay_epsilon*ay_mu)*n1x(xp(1),xp(2))*Sin(xp(2)))*xp(1))*(ay_I + ay_k2*Sqrt(&
&S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + x&
&p(1)**2)))/(ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2y1x = (/ REAL(LL2y1x,wp), AIMAG(LL2y1x) /)    
  END FUNCTION AY_LL2Y1X
!*****************************************************************************************************************************************

  FUNCTION ay_LL2y1y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2y1y(m)
    COMPLEX(wp) :: LL2y1y

    LL2y1y = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*n1z(xp(1),xp(2))*S1(xp(1),xp(2)) - ay_epsilon*n1x(x&
&p(1),xp(2))*Cos(x(2))*x(1) - n1y(xp(1),xp(2))*Sin(x(2))*x(1) - ay_epsilon*n1z(xp(1),xp(2))*x(3) + ay&
&_epsilon*n1x(xp(1),xp(2))*Cos(xp(2))*xp(1) + n1y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2y1y = (/ REAL(LL2y1y,wp), AIMAG(LL2y1y) /)    
  END FUNCTION AY_LL2Y1Y
!*****************************************************************************************************************************************

  FUNCTION ay_LL2y1z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2y1z(m)
    COMPLEX(wp) :: LL2y1z

    LL2y1z = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n1y(xp(1),xp(2))*(S1(xp(1),xp(2)) - x(&
&3)) + (-1 + ay_epsilon*ay_mu)*n1z(xp(1),xp(2))*(Sin(x(2))*x(1) - Sin(xp(2))*xp(1)))*(1 - ay_I*ay_k2*&
&Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1&
&) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**&
&2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2y1z = (/ REAL(LL2y1z,wp), AIMAG(LL2y1z) /)    
  END FUNCTION AY_LL2Y1Z
!*****************************************************************************************************************************************

  FUNCTION ay_LL2y2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2y2x(m)
    COMPLEX(wp) :: LL2y2x

    LL2y2x = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n2y(xp(1),xp(2))*Cos(x(2))*x(&
&1) - (-1 + ay_epsilon*ay_mu)*n2x(xp(1),xp(2))*Sin(x(2))*x(1) + (-(ay_epsilon*(-1 + ay_mu)*n2y(xp(1),&
&xp(2))*Cos(xp(2))) + (-1 + ay_epsilon*ay_mu)*n2x(xp(1),xp(2))*Sin(xp(2)))*xp(1))*(ay_I + ay_k2*Sqrt(&
&S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + x&
&p(1)**2)))/(ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2y2x = (/ REAL(LL2y2x,wp), AIMAG(LL2y2x) /)    
  END FUNCTION AY_LL2Y2X
!*****************************************************************************************************************************************

  FUNCTION ay_LL2y2y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2y2y(m)
    COMPLEX(wp) :: LL2y2y

    LL2y2y = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*n2z(xp(1),xp(2))*S2(xp(1),xp(2)) - ay_epsilon*n2x(x&
&p(1),xp(2))*Cos(x(2))*x(1) - n2y(xp(1),xp(2))*Sin(x(2))*x(1) - ay_epsilon*n2z(xp(1),xp(2))*x(3) + ay&
&_epsilon*n2x(xp(1),xp(2))*Cos(xp(2))*xp(1) + n2y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2y2y = (/ REAL(LL2y2y,wp), AIMAG(LL2y2y) /)    
  END FUNCTION AY_LL2Y2Y
!*****************************************************************************************************************************************

  FUNCTION ay_LL2y2z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2y2z(m)
    COMPLEX(wp) :: LL2y2z

    LL2y2z = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(ay_epsilon*(-1 + ay_mu)*n2y(xp(1),xp(2))*(S2(xp(1),xp(2)) - x(&
&3)) + (-1 + ay_epsilon*ay_mu)*n2z(xp(1),xp(2))*(Sin(x(2))*x(1) - Sin(xp(2))*xp(1)))*(1 - ay_I*ay_k2*&
&Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1&
&) + xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**&
&2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2y2z = (/ REAL(LL2y2z,wp), AIMAG(LL2y2z) /)    
  END FUNCTION AY_LL2Y2Z
!*****************************************************************************************************************************************

  FUNCTION ay_LL2z1x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2z1x(m)
    COMPLEX(wp) :: LL2z1x

    LL2z1x = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n1x(xp(1),xp(2))*S1(xp(1),xp(&
&2)) + ay_epsilon*(-1 + ay_mu)*n1z(xp(1),xp(2))*Cos(x(2))*x(1) + n1x(xp(1),xp(2))*x(3) - ay_epsilon*a&
&y_mu*n1x(xp(1),xp(2))*x(3) + ay_epsilon*n1z(xp(1),xp(2))*Cos(xp(2))*xp(1) - ay_epsilon*ay_mu*n1z(xp(&
&1),xp(2))*Cos(xp(2))*xp(1))*(ay_I + ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3)&
& + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(&
&1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2z1x = (/ REAL(LL2z1x,wp), AIMAG(LL2z1x) /)    
  END FUNCTION AY_LL2Z1X
!*****************************************************************************************************************************************

  FUNCTION ay_LL2z1y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2z1y(m)
    COMPLEX(wp) :: LL2z1y

    LL2z1y = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n1y(xp(1),xp(2))*S1(xp(1),xp(&
&2)) + ay_epsilon*(-1 + ay_mu)*n1z(xp(1),xp(2))*Sin(x(2))*x(1) + n1y(xp(1),xp(2))*x(3) - ay_epsilon*a&
&y_mu*n1y(xp(1),xp(2))*x(3) + ay_epsilon*n1z(xp(1),xp(2))*Sin(xp(2))*xp(1) - ay_epsilon*ay_mu*n1z(xp(&
&1),xp(2))*Sin(xp(2))*xp(1))*(ay_I + ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3)&
& + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(&
&1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)
         
    ay_LL2z1y = (/ REAL(LL2z1y,wp), AIMAG(LL2z1y) /)    
  END FUNCTION AY_LL2Z1Y
!*****************************************************************************************************************************************

  FUNCTION ay_LL2z1z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2z1z(m)
    COMPLEX(wp) :: LL2z1z

    LL2z1z = &
&(Exp(ay_I*ay_k2*Sqrt(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(n1z(xp(1),xp(2))*S1(xp(1),xp(2)) - ay_epsilon*n1x(xp(1),xp(2))&
&*Cos(x(2))*x(1) - ay_epsilon*n1y(xp(1),xp(2))*Sin(x(2))*x(1) - n1z(xp(1),xp(2))*x(3) + ay_epsilon*n1&
&x(xp(1),xp(2))*Cos(xp(2))*xp(1) + ay_epsilon*n1y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S1(xp(1),xp(2))**2 + x(1)**2 - 2*S1(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2z1z = (/ REAL(LL2z1z,wp), AIMAG(LL2z1z) /)    
  END FUNCTION AY_LL2Z1Z
!*****************************************************************************************************************************************

  FUNCTION ay_LL2z2x(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2z2x(m)
    COMPLEX(wp) :: LL2z2x

    LL2z2x = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n2x(xp(1),xp(2))*S2(xp(1),xp(&
&2)) + ay_epsilon*(-1 + ay_mu)*n2z(xp(1),xp(2))*Cos(x(2))*x(1) + n2x(xp(1),xp(2))*x(3) - ay_epsilon*a&
&y_mu*n2x(xp(1),xp(2))*x(3) + ay_epsilon*n2z(xp(1),xp(2))*Cos(xp(2))*xp(1) - ay_epsilon*ay_mu*n2z(xp(&
&1),xp(2))*Cos(xp(2))*xp(1))*(ay_I + ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3)&
& + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(&
&1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2z2x = (/ REAL(LL2z2x,wp), AIMAG(LL2z2x) /)    
  END FUNCTION AY_LL2Z2X
!*****************************************************************************************************************************************

  FUNCTION ay_LL2z2y(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2z2y(m)
    COMPLEX(wp) :: LL2z2y

    LL2z2y = &
&((0,0.25)*Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Co&
&s(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*((-1 + ay_epsilon*ay_mu)*n2y(xp(1),xp(2))*S2(xp(1),xp(&
&2)) + ay_epsilon*(-1 + ay_mu)*n2z(xp(1),xp(2))*Sin(x(2))*x(1) + n2y(xp(1),xp(2))*x(3) - ay_epsilon*a&
&y_mu*n2y(xp(1),xp(2))*x(3) + ay_epsilon*n2z(xp(1),xp(2))*Sin(xp(2))*xp(1) - ay_epsilon*ay_mu*n2z(xp(&
&1),xp(2))*Sin(xp(2))*xp(1))*(ay_I + ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3)&
& + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)))/(ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(&
&1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)

    ay_LL2z2y = (/ REAL(LL2z2y,wp), AIMAG(LL2z2y) /)    
  END FUNCTION AY_LL2Z2Y
!*****************************************************************************************************************************************

  FUNCTION ay_LL2z2z(xp,m)
    IMPLICIT NONE
    REAL(wp), INTENT(IN) :: xp(:)
    INTEGER, INTENT(IN) :: m
    REAL(wp) :: ay_LL2z2z(m)
    COMPLEX(wp) :: LL2z2z

    LL2z2z = &
&(Exp(ay_I*ay_k2*Sqrt(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - &
&xp(2))*x(1)*xp(1) + xp(1)**2))*xp(1)*(n2z(xp(1),xp(2))*S2(xp(1),xp(2)) - ay_epsilon*n2x(xp(1),xp(2))&
&*Cos(x(2))*x(1) - ay_epsilon*n2y(xp(1),xp(2))*Sin(x(2))*x(1) - n2z(xp(1),xp(2))*x(3) + ay_epsilon*n2&
&x(xp(1),xp(2))*Cos(xp(2))*xp(1) + ay_epsilon*n2y(xp(1),xp(2))*Sin(xp(2))*xp(1))*(1 - ay_I*ay_k2*Sqrt&
&(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - 2*Cos(x(2) - xp(2))*x(1)*xp(1) + &
&xp(1)**2)))/(4.*ay_epsilon*ay_Pi*(S2(xp(1),xp(2))**2 + x(1)**2 - 2*S2(xp(1),xp(2))*x(3) + x(3)**2 - &
&2*Cos(x(2) - xp(2))*x(1)*xp(1) + xp(1)**2)**1.5)
         
    ay_LL2z2z = (/ REAL(LL2z2z,wp), AIMAG(LL2z2z) /)    
  END FUNCTION AY_LL2Z2Z
!*****************************************************************************************************************************************
END MODULE ay_kernels
