!-----------------------------------------------------------------------------------------------------------------------------------------
! TITLE: ay_mat_eval.f95
! AUTHOR: Alex Yuffa
! DATE WRITTEN: 12/30/04
! LAST REVISION: 
! DESCRIPTION:  Computes E field OFF THE BOUNDARY
!-----------------------------------------------------------------------------------------------------------------------------------------

!-----------------------------------------------------------------------------------------------------------------------------------------
! NON-LOCAL SUBROUTINES USED: 
! MODULE NAME:          SUBROUTINE NAME:                                       COMMENT:
!                      
! NON-LOCAL FUNCTIONS USED:
! MODULE NAME:            FUNCTION NAME:                                       COMMENT:
!
! NON-LOCAL VARIABLES USED:
! MODULE NAME:            VARIABLE NAME:      TYPE:                             COMMENT:
!-----------------------------------------------------------------------------------------------------------------------------------------

!-----------------------------------------------------------------------------------------------------------------------------------------
! LOCAL VARIABLES USED:
! VARIABLE NAME:              TYPE:                    COMMENT:
!-----------------------------------------------------------------------------------------------------------------------------------------
MODULE ay_mat_eval
  
  USE ay_kind,         ONLY : wp
  USE ay_constants,    ONLY : ay_ko, ay_I
  USE ay_kernels,      ONLY : ay_write_obs_pt, ay_UL1x1x, ay_UL1x2x, ay_UR1x1x, ay_UR1x2x
  USE ay_integration,  ONLY : ay_sample_rho_p, ay_sample_theta_p, &
                              ay_tot_num_obs_pts, ay_n_rho_p, ay_n_theta_p, ay_obs_pts
  USE nag_quad_md,     ONLY : nag_quad_md_rect_mintg

  IMPLICIT NONE
CONTAINS
  SUBROUTINE ay_mat_eval_fill(obs_pt,bs,BD_values,field_value) 

    IMPLICIT NONE
    REAL(wp),    INTENT(IN) :: obs_pt(3)
    INTEGER,    INTENT(IN) :: bs
    COMPLEX(wp), INTENT(IN) :: BD_values(:)
    COMPLEX(wp), INTENT(OUT) :: field_value(3)
    
    !LOCAL
    INTEGER, PARAMETER :: max_fun_eval = 50000 !5000 !3400
    REAL(wp), PARAMETER :: rel_acc = 1.0e-10_wp !07/10/05 1.0e-8_wp  !1.0e-6_wp!1.0e-7_wp   !(epsilon(1.0_wp))**.25_wp
    REAL(wp), PARAMETER :: abs_acc = 1.0e-10_wp ! 07/10/05 1.0e-6_wp  !1.0e-16_wp
    INTEGER :: i_rho_p, i_theta_p, n_rho_p, n_theta_p
    REAL(wp), ALLOCATABLE :: rho_p(:), theta_p(:)
    REAL(wp) :: rho_mid_p
    REAL(wp) :: a(2), b(2) 
    REAL(wp) :: result_UL1x1x(2), result_UL1x2x(2), result_UR1x1x(2), result_UR1x2x(2)
    COMPLEX(wp), ALLOCATABLE :: UL1x1x(:), UL1x2x(:), UR1x1x(:), UR1x2x(:)
    INTEGER :: column_index

 
    
    !---ALLOCATING MEMORY
    ALLOCATE(UL1x1x(bs))
    ALLOCATE(UL1x2x(bs))    
    ALLOCATE(UR1x1x(bs))
    ALLOCATE(UR1x2x(bs))
    
    !--------------------------------------------------Integration-------------------------------------------------------
       CALL ay_write_obs_pt(obs_pt) !Writes observation points so it can be used by the integral kernel(s)


       column_index = 1       
       
       !**************************Set up sampling in rho prime
       n_rho_p = ay_n_rho_p() + 1
       ALLOCATE(rho_p(n_rho_p))
       CALL ay_sample_rho_p(rho_p)
       
       !***************************DOUBLE INTEGRAL
       do i_rho_p=1, n_rho_p - 1
 
          a(1) = rho_p(i_rho_p)
          b(1) = rho_p(i_rho_p+1)
          
          rho_mid_p = (b(1)+a(1))/2.0_wp
          n_theta_p = ay_n_theta_p(rho_mid_p) + 1
          ALLOCATE(theta_p(n_theta_p))
          CALL ay_sample_theta_p(rho_mid_p,theta_p)

          do i_theta_p=1, n_theta_p - 1

             a(2) = theta_p(i_theta_p)
             b(2) = theta_p(i_theta_p+1)
           
             !---Integrating UL1x1x
             CALL nag_quad_md_rect_mintg(ay_UL1x1x,a,b,result_UL1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
             UL1x1x(column_index) = cmplx(result_UL1x1x(1),result_UL1x1x(2),wp) 

             !---Integrating UR1x1x
             CALL nag_quad_md_rect_mintg(ay_UR1x1x,a,b,result_UR1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
             UR1x1x(column_index) = cmplx(result_UR1x1x(1),result_UR1x1x(2),wp) 

             !---Integrating UL1x2x
             CALL nag_quad_md_rect_mintg(ay_UL1x2x,a,b,result_UL1x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
             UL1x2x(column_index) = cmplx(result_UL1x2x(1),result_UL1x2x(2),wp)

             !---Integrating UR1x2x
             CALL nag_quad_md_rect_mintg(ay_UR1x2x,a,b,result_UR1x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
             UR1x2x(column_index) = cmplx(result_UR1x2x(1),result_UR1x2x(2),wp)
          
             column_index = column_index + 1
          end do
          DEALLOCATE(theta_p)
          
       end do !ending i_rho_p loop

       DEALLOCATE(rho_p)

       field_value(1) = SUM(UL1x1x(1:bs)*BD_values(1:bs)) + SUM(UL1x2x(1:bs)*BD_values(3*bs+1:4*bs)) + &
            SUM(UR1x1x(1:bs)*BD_values(6*bs+1:7*bs)) + SUM(UR1x2x(1:bs)*BD_values(9*bs+1:10*bs))

       field_value(2) = SUM(UL1x1x(1:bs)*BD_values(bs+1:2*bs)) + SUM(UL1x2x(1:bs)*BD_values(4*bs+1:5*bs)) + &
            SUM(UR1x1x(1:bs)*BD_values(7*bs+1:8*bs)) + SUM(UR1x2x(1:bs)*BD_values(10*bs+1:11*bs))

       field_value(3) = SUM(UL1x1x(:)*BD_values(2*bs+1:3*bs)) + SUM(UL1x2x(:)*BD_values(5*bs+1:6*bs)) + &
            SUM(UR1x1x(1:bs)*BD_values(8*bs+1:9*bs)) + SUM(UR1x2x(1:bs)*BD_values(11*bs+1:12*bs))  

    
    !---DEALOCATING MEMORY
    DEALLOCATE(UL1x1x)
    DEALLOCATE(UL1x2x)    
    DEALLOCATE(UR1x1x)
    DEALLOCATE(UR1x2x)
    
  END SUBROUTINE AY_MAT_EVAL_FILL
  
END MODULE AY_MAT_EVAL
