!-----------------------------------------------------------------------------------------------------------------------------------------
! TITLE: ay_matrix.f95
! AUTHOR: Alex Yuffa
! DATE WRITTEN: 08/09/04
! LAST REVISION: 07/09/05
! DESCRIPTION:  Driver for integral equations
!-----------------------------------------------------------------------------------------------------------------------------------------

!-----------------------------------------------------------------------------------------------------------------------------------------
! NON-LOCAL SUBROUTINES USED: 
! MODULE NAME:          SUBROUTINE NAME:                                       COMMENT:
! ay_kernels            ay_write_obs_pt( x(:) )                                Writes observation point so integral equations can use it.
! ay_integration        ay_sample_rho_p(n_rho_p, h_rho_p)                      Determines sampling in rho prime.
! ay_integration        ay_sample_theta_p(rho_mid_p, n_theta_p, h_theta_p)     Determines sampling in theta prime.  WARNING: This 
!                                                                              subroutine must always be called after ay_sample_rho_p
!                                                                              subroutine
! ay_integration        ay_num_sample_pts(num_sample_pts)                      Determines total number of sample points.
! ay_integration        ay_observation_pts( x(:,:), surf_num )                 Determines and writes observation points into an array
!                                                                              x(:,:).  Surface 1 -> surf_num=1, Surface 2 -> surf_num=2
! nag_quad_md           nag_quad_md_rect(f, a, b, &                            Computes a double integral of real f.  Note: [ ] denotes
!                       [rel_acc, rel_err, max_fun_eval, &                     optional arguments. See Nag's documentation for details.
!                        num_fun_eval, error]
!                      
! NON-LOCAL FUNCTIONS USED:
! MODULE NAME:            FUNCTION NAME:                                       COMMENT:
! ay_integration          ay_distance_cyl( x(:), xp(:) )                       Computes distance between x and xp in cylindrical coord.
!                                                                              system
!
! NON-LOCAL VARIABLES USED:
! MODULE NAME:            VARIABLE NAME:      TYPE:                             COMMENT:
! ay_kind                 wp                  INTEGER, PARAMETER                Working precision
! ay_constants            ay_R                REAL, PARAMETER                   Radius of the sphere
! ay_constants            ay_spw              INTEGER                           Number of sample points per wavelength
!-----------------------------------------------------------------------------------------------------------------------------------------

!-----------------------------------------------------------------------------------------------------------------------------------------
! LOCAL VARIABLES USED:
! VARIABLE NAME:              TYPE:                    COMMENT:
! max_fun_eval                INTEGER                  Maximum number of function evaluation to be used by Nag's integration routine.
!                                                      Note: max_fun_eval >= 17, Nag's default for max_fun_eval is 3400.
! rel_acc                     REAL                     Relative accuracy of an integral to be used by Nag's integration routine.
!                                                      Nag's default is (epsilon(1.0_wp))**(1/4)
! from_sing                   REAL                     Distance away from the singularity.
! num_sample_pts              INTEGER                  Number of observation (sample) points used in discretization of the surface.
! i_rho_p                     INTEGER                  Dummy iteration index for rho prime
! i_theta_p                   INTEGER                  Dummy iteration index for theta prime
! n_rho_p                     INTEGER                  Number of sample points in rho prime
! n_theta_p                   INTEGER                  Number of sample points in theta prime
! h_rho_p                     REAL                     Distance between two consecutive rho prime points
! h_theta_p                   REAL                     Distance between two consecutive theta prime points
! rho_mid_p                   RAEAL                    Average rho prime to be used in theta prime sampling
! x                           REAL, ARRAY              Observation point
! xp_surf1                    REAL, ARRAY              Integration point on surface one.
! xp_surf2                    REAL, ARRAY              Integration point on surface two.
! a                           REAL, ARRAY              Lower integration limit. a(1) = rho prime, a(2) = theta prime
! b                           REAL, ARRAY              Upper integration limit. b(1) = rho prime, b(2) = theta prime
! sing_a                      REAL, ARRAY              Lower integration limit to be used around the singularity. 
!                                                      sing_a(1) = rho prime, sing_a(2) = theta prime
! sing_b                      REAL, ARRAY              Upper integration limit to be used around the singularity.
!                                                      sing_b(1) = rho prime, sing_b(2) = theta prime
! x_mid_pts_S1                REAL, ARRAY              Where observation points for Surface 1 will be stored.  
!                                                      x_mid_pts_S1(1,:) = first observation point, x_mid_pts_S1(2,:) = second observation
!                                                      point, etc.
! x_mid_pts_S2                REAL, ARRAY              Where observation points for Surface 2 will be stored.  
!                                                      x_mid_pts_S2(1,:) = first observation point, x_mid_pts_S2(2,:) = second observation
!                                                      point, etc.
! i_mid_pts                   INTEGER                  Dummy iteration index for x_mid_pts
!-----------------------------------------------------------------------------------------------------------------------------------------

MODULE ay_matrix

  USE ay_kind,         ONLY : wp
  USE ay_constants,    ONLY : ay_R, ay_spw, ay_ko, ay_I, lambda_o, ay_pi
  USE ay_kernels,      ONLY : ay_write_obs_pt, ay_UL1x1x, ay_UL1x2x, ay_UR1x1x, ay_UR1x2x, ay_LR1x1x, ay_LR1x2x, &
                              ay_UL2x1x, ay_UL2x2x, ay_UR2x1x, ay_UR2x2x, ay_LR2x1x, ay_LR2x2x, ay_LL1x1x, &
                              ay_LL1x1y, ay_LL1x1z, ay_LL1x2x, ay_LL1x2y, ay_LL1x2z, &
                              ay_LL1y1x, ay_LL1y1y, ay_LL1y1z, ay_LL1y2x, ay_LL1y2y, ay_LL1y2z, &
                              ay_LL1z1x, ay_LL1z1y, ay_LL1z1z, ay_LL1z2x, ay_LL1z2y, ay_LL1z2z, &
                              ay_LL2x1x, ay_LL2x1y, ay_LL2x1z, ay_LL2y1x, ay_LL2y1y, ay_LL2y1z, &
                              ay_LL2z1x, ay_LL2z1y, ay_LL2z1z, &
                              ay_LL2x2x, ay_LL2x2y, ay_LL2x2z, ay_LL2y2x, ay_LL2y2y, ay_LL2y2z, &
                              ay_LL2z2x, ay_LL2z2y, ay_LL2z2z
  USE ay_integration,  ONLY : ay_distance_cyl, ay_sample_rho_p, ay_sample_theta_p, &
                              ay_tot_num_obs_pts, ay_n_rho_p, ay_n_theta_p, S1, S2, ay_obs_pts, &
                              c11_1, c12_1, c13_1, c22_1, c23_1, c33_1, &
                              c11_2, c12_2, c13_2, c22_2, c23_2, c33_2
  USE nag_quad_md,     ONLY : nag_quad_md_rect_mintg

IMPLICIT NONE
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_UL1x1x(:,:), data_UL1x2x(:,:), data_UL2x1x(:,:), data_UL2x2x(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_UR1x1x(:,:), data_UR1x2x(:,:), data_UR2x1x(:,:), data_UR2x2x(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LR1x1x(:,:), data_LR1x2x(:,:), data_LR2x1x(:,:), data_LR2x2x(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL1x1x(:,:), data_LL1x1y(:,:), data_LL1x1z(:,:), data_LL1x2x(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL1x2y(:,:), data_LL1x2z(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL1y1x(:,:), data_LL1y1y(:,:), data_LL1y1z(:,:), data_LL1y2x(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL1y2y(:,:), data_LL1y2z(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL1z1x(:,:), data_LL1z1y(:,:), data_LL1z1z(:,:), data_LL1z2x(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL1z2y(:,:), data_LL1z2z(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL2x1x(:,:), data_LL2x1y(:,:), data_LL2x1z(:,:), data_LL2x2x(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL2x2y(:,:), data_LL2x2z(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL2y1x(:,:), data_LL2y1y(:,:), data_LL2y1z(:,:), data_LL2y2x(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL2y2y(:,:), data_LL2y2z(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL2z1x(:,:), data_LL2z1y(:,:), data_LL2z1z(:,:), data_LL2z2x(:,:)
COMPLEX(wp), PUBLIC, SAVE, ALLOCATABLE :: data_LL2z2y(:,:), data_LL2z2z(:,:)

CONTAINS
SUBROUTINE ay_matrix_fill() 

  IMPLICIT NONE
  INTEGER, PARAMETER :: max_fun_eval = 50000 
  REAL(wp), PARAMETER :: sing_rho_p_scale = 1.0e-3_wp   ! 07/10/05 1.0e-2_wp 
  REAL(wp), PARAMETER :: sing_theta_p_scale = 1.0e-3_wp ! 07/10/05  1.0e-2_wp 
  REAL(wp), PARAMETER :: rel_acc = 1.0e-6_wp  ! 07/10/05 1.0e-6_wp was 1.0e-5_wp 
  REAL(wp), PARAMETER :: abs_acc = 1.0e-6_wp  ! 07/10/05 1.0e-6_wp was 1.0e-4_wp
  REAL(wp) :: from_sing, from_sing_theta 
  INTEGER :: num_sample_pts
  INTEGER :: i_rho_p, i_theta_p, n_rho_p, n_theta_p
  REAL(wp), ALLOCATABLE :: rho_p(:), theta_p(:)
  REAL(wp) :: rho_mid_p, diff_rho_p
  REAL(wp) :: x(3), xp_surf1(3), xp_surf2(3)
  REAL(wp) :: a(2), b(2), sing_a(2), sing_b(2)
  REAL(wp) :: m_sing_a(2), m_sing_b(2)
  INTEGER :: i_mid_pts
  REAL(wp) :: result_UL1x1x(2), result_UL1x2x(2), result_UR1x1x(2), result_UR1x2x(2), result_LR1x1x(2), result_LR1x2x(2)
  REAL(wp) :: sing_result_UL1x1x(2), sing_result_UR1x1x(2), sing_result_LR1x1x(2)
  REAL(wp) :: result_UL2x1x(2), result_UL2x2x(2), result_UR2x1x(2), result_UR2x2x(2), result_LR2x1x(2), result_LR2x2x(2)
  REAL(wp) :: sing_result_UL2x2x(2), sing_result_UR2x2x(2), sing_result_LR2x2x(2)
  REAL(wp) :: result_LL1x1x(2), result_LL1x1y(2), result_LL1x1z(2)
  REAL(wp) :: sing_result_LL1x1x(2), sing_result_LL1x1y(2), sing_result_LL1x1z(2)
  REAL(wp) :: result_LL1x2x(2), result_LL1x2y(2), result_LL1x2z(2)
  REAL(wp) :: result_LL1y1x(2), result_LL1y1y(2), result_LL1y1z(2)
  REAL(wp) :: sing_result_LL1y1x(2), sing_result_LL1y1y(2), sing_result_LL1y1z(2)
  REAL(wp) :: result_LL1z1x(2), result_LL1z1y(2), result_LL1z1z(2)
  REAL(wp) :: sing_result_LL1z1x(2), sing_result_LL1z1y(2), sing_result_LL1z1z(2)
  REAL(wp) :: result_LL1y2x(2), result_LL1y2y(2), result_LL1y2z(2), result_LL1z2x(2), result_LL1z2y(2), result_LL1z2z(2)
  REAL(wp) :: result_LL2x1x(2), result_LL2x1y(2), result_LL2x1z(2), result_LL2y1x(2), result_LL2y1y(2), result_LL2y1z(2)
  REAl(wp) :: result_LL2z1x(2), result_LL2z1y(2), result_LL2z1z(2)
  REAL(wp) :: result_LL2x2x(2), result_LL2x2y(2), result_LL2x2z(2), result_LL2y2x(2), result_LL2y2y(2), result_LL2y2z(2)
  REAL(wp) :: result_LL2z2x(2), result_LL2z2y(2), result_LL2z2z(2)
  REAL(wp) :: sing_result_LL2x2x(2), sing_result_LL2x2y(2), sing_result_LL2x2z(2)
  REAL(wp) :: sing_result_LL2y2x(2), sing_result_LL2y2y(2), sing_result_LL2y2z(2)
  REAL(wp) :: sing_result_LL2z2x(2), sing_result_LL2z2y(2), sing_result_LL2z2z(2)
  REAL(wp), ALLOCATABLE :: x_mid_pts_S1(:,:) !Dynamic allocation of memory.  Don't forget to DEALLOCATE x_mid_pts_S1
  REAL(wp), ALLOCATABLE :: x_mid_pts_S2(:,:) !Dynamic allocation of memory.  Don't forget to DEALLOCATE x_mid_ptsS2
  INTEGER :: column_index
  !****************************Estimates the time required by the program
  num_sample_pts = ay_tot_num_obs_pts() !Determines total number of obs pts/Surf

  WRITE(*,*) 'Number of nodes/surface = ', num_sample_pts

  
  !---ALLOCATING MEMORY
  ALLOCATE(data_UL1x1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_UL1x2x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_UL2x1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_UL2x2x(num_sample_pts,num_sample_pts))

  ALLOCATE(data_UR1x1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_UR1x2x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_UR2x1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_UR2x2x(num_sample_pts,num_sample_pts))

  ALLOCATE(data_LR1x1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LR1x2x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LR2x1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LR2x2x(num_sample_pts,num_sample_pts))

  ALLOCATE(data_LL1x1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1x1y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1x1z(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1x2x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1x2y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1x2z(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1y1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1y1y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1y1z(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1y2x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1y2y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1y2z(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1z1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1z1y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1z1z(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1z2x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1z2y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL1z2z(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2x1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2x1y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2x1z(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2x2x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2x2y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2x2z(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2y1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2y1y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2y1z(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2y2x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2y2y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2y2z(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2z1x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2z1y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2z1z(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2z2x(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2z2y(num_sample_pts,num_sample_pts))
  ALLOCATE(data_LL2z2z(num_sample_pts,num_sample_pts))

  !****************************Allocating memory and writing observation points into an array
  ALLOCATE( x_mid_pts_S1(num_sample_pts,3) ) !Allocating memory for an array of obs. pts
  CALL ay_obs_pts(x_mid_pts_S1,1) !Writes observation mid. pts. for surface 1 into x_mid_pts_S1

  !--------------------------------------------------Obsevation Points on Surface 1-------------------------------------------------------
  do i_mid_pts=1, num_sample_pts
     x = x_mid_pts_S1(i_mid_pts,:)
     CALL ay_write_obs_pt(x) !Writes observation points so it can be used by the integral kernel(s)
     
     column_index = 1

     !**************************Set up sampling in rho prime
     n_rho_p = ay_n_rho_p() + 1
     ALLOCATE(rho_p(n_rho_p))
     CALL ay_sample_rho_p(rho_p)

     !***************************DOUBLE INTEGRAL
     do i_rho_p=1, n_rho_p - 1
        
        a(1) = rho_p(i_rho_p)
        b(1) = rho_p(i_rho_p+1)
        
     rho_mid_p = (b(1)+a(1))/2.0_wp
     n_theta_p = ay_n_theta_p(rho_mid_p) + 1
     ALLOCATE(theta_p(n_theta_p))
     CALL ay_sample_theta_p(rho_mid_p,theta_p)

        do i_theta_p=1, n_theta_p - 1

           !********************Computing mid points for the integration pts on Surface 1 that are used to find the singularity
           xp_surf1(1) = rho_mid_p
           xp_surf1(2) = (theta_p(i_theta_p+1) + theta_p(i_theta_p))/2.0_wp
           xp_surf1(3) = S1(rho_mid_p, xp_surf1(2) )

           a(2) = theta_p(i_theta_p)
           b(2) = theta_p(i_theta_p+1)
           diff_rho_p = rho_p(i_mid_pts+1)-rho_p(i_mid_pts)

           if( ay_distance_cyl(x,xp_surf1) > epsilon(diff_rho_p) ) then !Not near the singularity

              !---Integrating UL1x1x
              CALL nag_quad_md_rect_mintg(ay_UL1x1x,a,b,result_UL1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_UL1x1x(i_mid_pts,column_index) = cmplx(result_UL1x1x(1),result_UL1x1x(2),wp)

              !---Integrating UR1x1x
              CALL nag_quad_md_rect_mintg(ay_UR1x1x,a,b,result_UR1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_UR1x1x(i_mid_pts,column_index) = cmplx(result_UR1x1x(1),result_UR1x1x(2),wp)

              !---Integrating LR1x1x
              CALL nag_quad_md_rect_mintg(ay_LR1x1x,a,b,result_LR1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LR1x1x(i_mid_pts,column_index) = cmplx(result_LR1x1x(1),result_LR1x1x(2),wp)

              !---Integrating LL1x1x
              CALL nag_quad_md_rect_mintg(ay_LL1x1x,a,b,result_LL1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL1x1x(i_mid_pts,column_index) = cmplx(result_LL1x1x(1),result_LL1x1x(2),wp)

              !---Integrating LL1x1y
              CALL nag_quad_md_rect_mintg(ay_LL1x1y,a,b,result_LL1x1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL1x1y(i_mid_pts,column_index) = cmplx(result_LL1x1y(1),result_LL1x1y(2),wp)
              
              !---Integrating LL1x1z
              CALL nag_quad_md_rect_mintg(ay_LL1x1z,a,b,result_LL1x1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL1x1z(i_mid_pts,column_index) = cmplx(result_LL1x1z(1),result_LL1x1z(2),wp)

              !---Integrating LL1y1x
              CALL nag_quad_md_rect_mintg(ay_LL1y1x,a,b,result_LL1y1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL1y1x(i_mid_pts,column_index) = cmplx(result_LL1y1x(1),result_LL1y1x(2),wp)

              !---Integrating LL1y1y
              CALL nag_quad_md_rect_mintg(ay_LL1y1y,a,b,result_LL1y1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL1y1y(i_mid_pts,column_index) = cmplx(result_LL1y1y(1),result_LL1y1y(2),wp)

              !---Integrating LL1y1z
              CALL nag_quad_md_rect_mintg(ay_LL1y1z,a,b,result_LL1y1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL1y1z(i_mid_pts,column_index) = cmplx(result_LL1y1z(1),result_LL1y1z(2),wp)

              !---Integrating LL1z1x
              CALL nag_quad_md_rect_mintg(ay_LL1z1x,a,b,result_LL1z1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL1z1x(i_mid_pts,column_index) = cmplx(result_LL1z1x(1),result_LL1z1x(2),wp)

              !---Integrating LL1z1y
              CALL nag_quad_md_rect_mintg(ay_LL1z1y,a,b,result_LL1z1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL1z1y(i_mid_pts,column_index) = cmplx(result_LL1z1y(1),result_LL1z1y(2),wp)

              !---Integrating LL1z1z
              CALL nag_quad_md_rect_mintg(ay_LL1z1z,a,b,result_LL1z1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL1z1z(i_mid_pts,column_index) = cmplx(result_LL1z1z(1),result_LL1z1z(2),wp)

           else !Near the singularity


              from_sing = abs(b(1)-a(1))*sing_rho_p_scale !Distance away from singularity in rho
              from_sing_theta = abs(b(2)-a(2))*sing_theta_p_scale !Distance away from singularity in theta

              !***********************Integrating to the Left of the Singularity
              sing_b(1) = xp_surf1(1) - from_sing
              sing_b(2) = b(2)
              
              !---Integrating UL1x1x
              CALL nag_quad_md_rect_mintg(ay_UL1x1x,a,sing_b,result_UL1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UL1x1x = result_UL1x1x

              !---Integrating UR1x1x
              CALL nag_quad_md_rect_mintg(ay_UR1x1x,a,sing_b,result_UR1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UR1x1x = result_UR1x1x

              !---Integrating LR1x1x
              CALL nag_quad_md_rect_mintg(ay_LR1x1x,a,sing_b,result_LR1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LR1x1x = result_LR1x1x
             
              !---Integrating LL1x1x
              CALL nag_quad_md_rect_mintg(ay_LL1x1x,a,sing_b,result_LL1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1x1x = result_LL1x1x

              !---Integrating LL1x1y
              CALL nag_quad_md_rect_mintg(ay_LL1x1y,a,sing_b,result_LL1x1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1x1y = result_LL1x1y

              !---Integrating LL1x1z
              CALL nag_quad_md_rect_mintg(ay_LL1x1z,a,sing_b,result_LL1x1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1x1z = result_LL1x1z

              !---Integrating LL1y1x
              CALL nag_quad_md_rect_mintg(ay_LL1y1x,a,sing_b,result_LL1y1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1y1x = result_LL1y1x

              !---Integrating LL1y1y
              CALL nag_quad_md_rect_mintg(ay_LL1y1y,a,sing_b,result_LL1y1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1y1y = result_LL1y1y

              !---Integrating LL1y1z
              CALL nag_quad_md_rect_mintg(ay_LL1y1z,a,sing_b,result_LL1y1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1y1z = result_LL1y1z

              !---Integrating LL1z1x
              CALL nag_quad_md_rect_mintg(ay_LL1z1x,a,sing_b,result_LL1z1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1z1x = result_LL1z1x

              !---Integrating LL1z1y
              CALL nag_quad_md_rect_mintg(ay_LL1z1y,a,sing_b,result_LL1z1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1z1y = result_LL1z1y

              !---Integrating LL1z1z
              CALL nag_quad_md_rect_mintg(ay_LL1z1z,a,sing_b,result_LL1z1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1z1z = result_LL1z1z

              !*************************************Lower Rectangle****************
              m_sing_a(1) = xp_surf1(1) - from_sing
              m_sing_b(1) = xp_surf1(1) + from_sing
              m_sing_a(2) = a(2)
              m_sing_b(2) = xp_surf1(2) - from_sing_theta
              
              !---Integrating UL1x1x
              CALL nag_quad_md_rect_mintg(ay_UL1x1x,m_sing_a,m_sing_b,result_UL1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UL1x1x = sing_result_UL1x1x + result_UL1x1x

              !---Integrating UR1x1x
              CALL nag_quad_md_rect_mintg(ay_UR1x1x,m_sing_a,m_sing_b,result_UR1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UR1x1x = sing_result_UR1x1x + result_UR1x1x

              !---Integrating LR1x1x
              CALL nag_quad_md_rect_mintg(ay_LR1x1x,m_sing_a,m_sing_b,result_LR1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LR1x1x = sing_result_LR1x1x + result_LR1x1x
             
              !---Integrating LL1x1x
              CALL nag_quad_md_rect_mintg(ay_LL1x1x,m_sing_a,m_sing_b,result_LL1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1x1x = sing_result_LL1x1x + result_LL1x1x

              !---Integrating LL1x1y
              CALL nag_quad_md_rect_mintg(ay_LL1x1y,m_sing_a,m_sing_b,result_LL1x1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1x1y = sing_result_LL1x1y + result_LL1x1y

              !---Integrating LL1x1z
              CALL nag_quad_md_rect_mintg(ay_LL1x1z,m_sing_a,m_sing_b,result_LL1x1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1x1z = sing_result_LL1x1z + result_LL1x1z

              !---Integrating LL1y1x
              CALL nag_quad_md_rect_mintg(ay_LL1y1x,m_sing_a,m_sing_b,result_LL1y1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1y1x = sing_result_LL1y1x + result_LL1y1x

              !---Integrating LL1y1y
              CALL nag_quad_md_rect_mintg(ay_LL1y1y,m_sing_a,m_sing_b,result_LL1y1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1y1y = sing_result_LL1y1y + result_LL1y1y

              !---Integrating LL1y1z
              CALL nag_quad_md_rect_mintg(ay_LL1y1z,m_sing_a,m_sing_b,result_LL1y1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1y1z = sing_result_LL1y1z + result_LL1y1z

              !---Integrating LL1z1x
              CALL nag_quad_md_rect_mintg(ay_LL1z1x,m_sing_a,m_sing_b,result_LL1z1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1z1x = sing_result_LL1z1x + result_LL1z1x

              !---Integrating LL1z1y
              CALL nag_quad_md_rect_mintg(ay_LL1z1y,m_sing_a,m_sing_b,result_LL1z1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1z1y = sing_result_LL1z1y + result_LL1z1y

              !---Integrating LL1z1z
              CALL nag_quad_md_rect_mintg(ay_LL1z1z,m_sing_a,m_sing_b,result_LL1z1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1z1z = sing_result_LL1z1z + result_LL1z1z

              !*********************************************Upper Rectangle******************************
              m_sing_a(1) = xp_surf1(1) - from_sing
              m_sing_b(1) = xp_surf1(1) + from_sing
              m_sing_a(2) = xp_surf1(2) + from_sing_theta
              m_sing_b(2) = b(2)
             
              !---Integrating UL1x1x
              CALL nag_quad_md_rect_mintg(ay_UL1x1x,m_sing_a,m_sing_b,result_UL1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UL1x1x = sing_result_UL1x1x + result_UL1x1x

              !---Integrating UR1x1x
              CALL nag_quad_md_rect_mintg(ay_UR1x1x,m_sing_a,m_sing_b,result_UR1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UR1x1x = sing_result_UR1x1x + result_UR1x1x

              !---Integrating LR1x1x
              CALL nag_quad_md_rect_mintg(ay_LR1x1x,m_sing_a,m_sing_b,result_LR1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LR1x1x = sing_result_LR1x1x + result_LR1x1x
             
              !---Integrating LL1x1x
              CALL nag_quad_md_rect_mintg(ay_LL1x1x,m_sing_a,m_sing_b,result_LL1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1x1x = sing_result_LL1x1x + result_LL1x1x

              !---Integrating LL1x1y
              CALL nag_quad_md_rect_mintg(ay_LL1x1y,m_sing_a,m_sing_b,result_LL1x1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1x1y = sing_result_LL1x1y + result_LL1x1y

              !---Integrating LL1x1z
              CALL nag_quad_md_rect_mintg(ay_LL1x1z,m_sing_a,m_sing_b,result_LL1x1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1x1z = sing_result_LL1x1z + result_LL1x1z

              !---Integrating LL1y1x
              CALL nag_quad_md_rect_mintg(ay_LL1y1x,m_sing_a,m_sing_b,result_LL1y1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1y1x = sing_result_LL1y1x + result_LL1y1x

              !---Integrating LL1y1y
              CALL nag_quad_md_rect_mintg(ay_LL1y1y,m_sing_a,m_sing_b,result_LL1y1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1y1y = sing_result_LL1y1y + result_LL1y1y

              !---Integrating LL1y1z
              CALL nag_quad_md_rect_mintg(ay_LL1y1z,m_sing_a,m_sing_b,result_LL1y1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1y1z = sing_result_LL1y1z + result_LL1y1z

              !---Integrating LL1z1x
              CALL nag_quad_md_rect_mintg(ay_LL1z1x,m_sing_a,m_sing_b,result_LL1z1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1z1x = sing_result_LL1z1x + result_LL1z1x

              !---Integrating LL1z1y
              CALL nag_quad_md_rect_mintg(ay_LL1z1y,m_sing_a,m_sing_b,result_LL1z1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1z1y = sing_result_LL1z1y + result_LL1z1y

              !---Integrating LL1z1z
              CALL nag_quad_md_rect_mintg(ay_LL1z1z,m_sing_a,m_sing_b,result_LL1z1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL1z1z = sing_result_LL1z1z + result_LL1z1z

              !***********************Integrating to the Right of the Singularity**************************
              sing_a(1) = xp_surf1(1) + from_sing
              sing_a(2) = a(2)

              !---Integrating UL1x1x
              CALL nag_quad_md_rect_mintg(ay_UL1x1x,sing_a,b,result_UL1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_UL1x1x = sing_result_UL1x1x + result_UL1x1x
              data_UL1x1x(i_mid_pts,column_index) = cmplx(result_UL1x1x(1)+.5_wp,result_UL1x1x(2),wp)
 
              !---Integrating UR1x1x
              CALL nag_quad_md_rect_mintg(ay_UR1x1x,sing_a,b,result_UR1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_UR1x1x = sing_result_UR1x1x + result_UR1x1x
              data_UR1x1x(i_mid_pts,column_index) = cmplx(result_UR1x1x(1),result_UR1x1x(2),wp)

              !---Integrating LR1x1x
              CALL nag_quad_md_rect_mintg(ay_LR1x1x,sing_a,b,result_LR1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LR1x1x = sing_result_LR1x1x + result_LR1x1x
              data_LR1x1x(i_mid_pts,column_index) = cmplx(result_LR1x1x(1),result_LR1x1x(2),wp)
              
              !---Integrating LL1x1x
              CALL nag_quad_md_rect_mintg(ay_LL1x1x,sing_a,b,result_LL1x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL1x1x = sing_result_LL1x1x + result_LL1x1x !c11
              data_LL1x1x(i_mid_pts,column_index) = cmplx(result_LL1x1x(1),result_LL1x1x(2),wp)-.5_wp*c11_1(x(1),x(2))
              
              !---Integrating LL1x1y
              CALL nag_quad_md_rect_mintg(ay_LL1x1y,sing_a,b,result_LL1x1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL1x1y = sing_result_LL1x1y + result_LL1x1y !c12
              data_LL1x1y(i_mid_pts,column_index) = cmplx(result_LL1x1y(1),result_LL1x1y(2),wp)-.5_wp*c12_1(x(1),x(2))
              
              !---Integrating LL1x1z
              CALL nag_quad_md_rect_mintg(ay_LL1x1z,sing_a,b,result_LL1x1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL1x1z = sing_result_LL1x1z + result_LL1x1z !c13
              data_LL1x1z(i_mid_pts,column_index) = cmplx(result_LL1x1z(1),result_LL1x1z(2),wp)-.5_wp*c13_1(x(1),x(2))

              !---Integrating LL1y1x
              CALL nag_quad_md_rect_mintg(ay_LL1y1x,sing_a,b,result_LL1y1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL1y1x = sing_result_LL1y1x + result_LL1y1x  !c12=c21
              data_LL1y1x(i_mid_pts,column_index) = cmplx(result_LL1y1x(1),result_LL1y1x(2),wp)-.5_wp*c12_1(x(1),x(2))

              !---Integrating LL1y1y
              CALL nag_quad_md_rect_mintg(ay_LL1y1y,sing_a,b,result_LL1y1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL1y1y = sing_result_LL1y1y + result_LL1y1y !c22
              data_LL1y1y(i_mid_pts,column_index) = cmplx(result_LL1y1y(1),result_LL1y1y(2),wp)-.5_wp*c22_1(x(1),x(2))

              !---Integrating LL1y1z
              CALL nag_quad_md_rect_mintg(ay_LL1y1z,sing_a,b,result_LL1y1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL1y1z = sing_result_LL1y1z + result_LL1y1z !c23
              data_LL1y1z(i_mid_pts,column_index) = cmplx(result_LL1y1z(1),result_LL1y1z(2),wp)-.5_wp*c23_1(x(1),x(2))

              !---Integrating LL1z1x
              CALL nag_quad_md_rect_mintg(ay_LL1z1x,sing_a,b,result_LL1z1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL1z1x = sing_result_LL1z1x + result_LL1z1x  !c13=c31
              data_LL1z1x(i_mid_pts,column_index) = cmplx(result_LL1z1x(1),result_LL1z1x(2),wp)-.5_wp*c13_1(x(1),x(2))

              !---Integrating LL1z1y
              CALL nag_quad_md_rect_mintg(ay_LL1z1y,sing_a,b,result_LL1z1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL1z1y = sing_result_LL1z1y + result_LL1z1y !c23=c32
              data_LL1z1y(i_mid_pts,column_index) = cmplx(result_LL1z1y(1),result_LL1z1y(2),wp)-.5_wp*c23_1(x(1),x(2))

              !---Integrating LL1z1z
              CALL nag_quad_md_rect_mintg(ay_LL1z1z,sing_a,b,result_LL1z1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL1z1z = sing_result_LL1z1z + result_LL1z1z !c33
              data_LL1z1z(i_mid_pts,column_index) = cmplx(result_LL1z1z(1),result_LL1z1z(2),wp)-.5_wp*c33_1(x(1),x(2))

           end if

           !---Integrating UL1x2x
           CALL nag_quad_md_rect_mintg(ay_UL1x2x,a,b,result_UL1x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_UL1x2x(i_mid_pts,column_index) = cmplx(result_UL1x2x(1),result_UL1x2x(2),wp) 

           !---Integrating UR1x2x
           CALL nag_quad_md_rect_mintg(ay_UR1x2x,a,b,result_UR1x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_UR1x2x(i_mid_pts,column_index) = cmplx(result_UR1x2x(1),result_UR1x2x(2),wp)

           !---Integrating LR1x2x
           CALL nag_quad_md_rect_mintg(ay_LR1x2x,a,b,result_LR1x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LR1x2x(i_mid_pts,column_index) = cmplx(result_LR1x2x(1),result_LR1x2x(2),wp)

           !---Integrating LL1x2x
           CALL nag_quad_md_rect_mintg(ay_LL1x2x,a,b,result_LL1x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL1x2x(i_mid_pts,column_index) = cmplx(result_LL1x2x(1),result_LL1x2x(2),wp)

           !---Integrating LL1x2y
           CALL nag_quad_md_rect_mintg(ay_LL1x2y,a,b,result_LL1x2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL1x2y(i_mid_pts,column_index) = cmplx(result_LL1x2y(1),result_LL1x2y(2),wp)

           !---Integrating LL1x2z
           CALL nag_quad_md_rect_mintg(ay_LL1x2z,a,b,result_LL1x2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL1x2z(i_mid_pts,column_index) = cmplx(result_LL1x2z(1),result_LL1x2z(2),wp)

           !---Integrating LL1y2x
           CALL nag_quad_md_rect_mintg(ay_LL1y2x,a,b,result_LL1y2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL1y2x(i_mid_pts,column_index) = cmplx(result_LL1y2x(1),result_LL1y2x(2),wp)

           !---Integrating LL1y2y
           CALL nag_quad_md_rect_mintg(ay_LL1y2y,a,b,result_LL1y2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL1y2y(i_mid_pts,column_index) = cmplx(result_LL1y2y(1),result_LL1y2y(2),wp)

           !---Integrating LL1y2z
           CALL nag_quad_md_rect_mintg(ay_LL1y2z,a,b,result_LL1y2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL1y2z(i_mid_pts,column_index) = cmplx(result_LL1y2z(1),result_LL1y2z(2),wp)

           !---Integrating LL1z2x
           CALL nag_quad_md_rect_mintg(ay_LL1z2x,a,b,result_LL1z2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL1z2x(i_mid_pts,column_index) = cmplx(result_LL1z2x(1),result_LL1z2x(2),wp)

           !---Integrating LL1z2y
           CALL nag_quad_md_rect_mintg(ay_LL1z2y,a,b,result_LL1z2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL1z2y(i_mid_pts,column_index) = cmplx(result_LL1z2y(1),result_LL1z2y(2),wp)

           !---Integrating LL1z2z
           CALL nag_quad_md_rect_mintg(ay_LL1z2z,a,b,result_LL1z2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL1z2z(i_mid_pts,column_index) = cmplx(result_LL1z2z(1),result_LL1z2z(2),wp)
           
           column_index = column_index+1
        end do
        DEALLOCATE(theta_p)
        
        if( b(1)>ay_R ) then !b(1) must be <= ay_R.  Due to round off errors b(1) may become larger than ay_R
           write(*,*) 'b(1) > ay_R, Error for Surface 1, aborting'
           stop
        end if
     end do !ending i_rho_p loop

     DEALLOCATE(rho_p)

  end do !endding i_mid_pts loop

  DEALLOCATE(x_mid_pts_S1) !Deallocating memory

  !-----------------------------------------Observation Point on Surface 2----------------------------------------------------------------


  !****************************Allocating memory and writing obs pts into an array
  ALLOCATE( x_mid_pts_S2(num_sample_pts,3) ) !Allocating memory for an array of obs pts
  CALL ay_obs_pts(x_mid_pts_S2,2) !Writes obs mid. pts. for surface 2 into x_mid_pts

  !--------------------------------------------------Obsevation Points on Surface 2-------------------------------------------------------
  do i_mid_pts=1, num_sample_pts
     x = x_mid_pts_S2(i_mid_pts,:)
     CALL ay_write_obs_pt(x) !Writes observation points so it can be used by the integral kernel(s)

     column_index = 1

     !**************************Set up sampling in rho prime
     n_rho_p = ay_n_rho_p() + 1
     ALLOCATE(rho_p(n_rho_p))
     CALL ay_sample_rho_p(rho_p)
     
     !***************************DOUBLE INTEGRAL
     do i_rho_p=1, n_rho_p - 1

        a(1) = rho_p(i_rho_p)
        b(1) = rho_p(i_rho_p+1)

        rho_mid_p = (b(1)+a(1))/2.0_wp
        n_theta_p = ay_n_theta_p(rho_mid_p) + 1
        ALLOCATE(theta_p(n_theta_p))
        CALL ay_sample_theta_p(rho_mid_p,theta_p)

        do i_theta_p=1, n_theta_p - 1
           
           !********************Computing mid points for the integration pts on Surface 2 that are used to find the singularity
           xp_surf2(1) = rho_mid_p
           xp_surf2(2) = (theta_p(i_theta_p+1) + theta_p(i_theta_p))/2.0_wp
           xp_surf2(3) = S2(rho_mid_p, xp_surf2(2) )
           
           a(2) = theta_p(i_theta_p)
           b(2) = theta_p(i_theta_p+1)   
           diff_rho_p =  rho_p(i_mid_pts+1)-rho_p(i_mid_pts)

           if( ay_distance_cyl(x,xp_surf2) > epsilon(diff_rho_p)) then !Not near the singularity

              !---Integrating UL2x2x
              CALL nag_quad_md_rect_mintg(ay_UL2x2x,a,b,result_UL2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_UL2x2x(i_mid_pts,column_index) = cmplx(result_UL2x2x(1),result_UL2x2x(2),wp)

              !---Integrating UR2x2x
              CALL nag_quad_md_rect_mintg(ay_UR2x2x,a,b,result_UR2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_UR2x2x(i_mid_pts,column_index) = cmplx(result_UR2x2x(1),result_UR2x2x(2),wp)

              !---Integrating LR2x2x
              CALL nag_quad_md_rect_mintg(ay_LR2x2x,a,b,result_LR2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LR2x2x(i_mid_pts,column_index) = cmplx(result_LR2x2x(1),result_LR2x2x(2),wp)

              !---Integrating LL2x2x
              CALL nag_quad_md_rect_mintg(ay_LL2x2x,a,b,result_LL2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL2x2x(i_mid_pts,column_index) = cmplx(result_LL2x2x(1),result_LL2x2x(2),wp)

              !---Integrating LL2x2y
              CALL nag_quad_md_rect_mintg(ay_LL2x2y,a,b,result_LL2x2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL2x2y(i_mid_pts,column_index) = cmplx(result_LL2x2y(1),result_LL2x2y(2),wp)

              !---Integrating LL2x2z
              CALL nag_quad_md_rect_mintg(ay_LL2x2z,a,b,result_LL2x2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL2x2z(i_mid_pts,column_index) = cmplx(result_LL2x2z(1),result_LL2x2z(2),wp)

              !---Integrating LL2y2x
              CALL nag_quad_md_rect_mintg(ay_LL2y2x,a,b,result_LL2y2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL2y2x(i_mid_pts,column_index) = cmplx(result_LL2y2x(1),result_LL2y2x(2),wp)

              !---Integrating LL2y2y
              CALL nag_quad_md_rect_mintg(ay_LL2y2y,a,b,result_LL2y2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL2y2y(i_mid_pts,column_index) = cmplx(result_LL2y2y(1),result_LL2y2y(2),wp)

              !---Integrating LL2y2z
              CALL nag_quad_md_rect_mintg(ay_LL2y2z,a,b,result_LL2y2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL2y2z(i_mid_pts,column_index) = cmplx(result_LL2y2z(1),result_LL2y2z(2),wp)

              !---Integrating LL2z2x
              CALL nag_quad_md_rect_mintg(ay_LL2z2x,a,b,result_LL2z2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL2z2x(i_mid_pts,column_index) = cmplx(result_LL2z2x(1),result_LL2z2x(2),wp)

              !---Integrating LL2z2y
              CALL nag_quad_md_rect_mintg(ay_LL2z2y,a,b,result_LL2z2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL2z2y(i_mid_pts,column_index) = cmplx(result_LL2z2y(1),result_LL2z2y(2),wp)

              !---Integrating LL2z2z
              CALL nag_quad_md_rect_mintg(ay_LL2z2z,a,b,result_LL2z2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
              data_LL2z2z(i_mid_pts,column_index) = cmplx(result_LL2z2z(1),result_LL2z2z(2),wp)

           else !Near the singularity

              from_sing = abs(b(1)-a(1))*sing_rho_p_scale
              from_sing_theta = abs(b(2)-a(2))*sing_theta_p_scale
              
              !***********************Integrating to the Left of the Singularity****************************
              sing_b(1) = xp_surf2(1) - from_sing
              sing_b(2) = b(2)

              !---Integrating UL2x2x
              CAll nag_quad_md_rect_mintg(ay_UL2x2x,a,sing_b,result_UL2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UL2x2x = result_UL2x2x

              !---Integrating UR2x2x
              CALL nag_quad_md_rect_mintg(ay_UR2x2x,a,sing_b,result_UR2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UR2x2x = result_UR2x2x

              !---Integrating LR2x2x
              CALL nag_quad_md_rect_mintg(ay_LR2x2x,a,sing_b,result_LR2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LR2x2x = result_LR2x2x

              !---Integrating LL2x2x
              CALL nag_quad_md_rect_mintg(ay_LL2x2x,a,sing_b,result_LL2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2x2x = result_LL2x2x

              !---Integrating LL2x2y
              CALL nag_quad_md_rect_mintg(ay_LL2x2y,a,sing_b,result_LL2x2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2x2y = result_LL2x2y

              !---Integrating LL2x2z
              CALL nag_quad_md_rect_mintg(ay_LL2x2z,a,sing_b,result_LL2x2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2x2z = result_LL2x2z

              !---Integrating LL2y2x
              CALL nag_quad_md_rect_mintg(ay_LL2y2x,a,sing_b,result_LL2y2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2y2x = result_LL2y2x

              !---Integrating LL2y2y
              CALL nag_quad_md_rect_mintg(ay_LL2y2y,a,sing_b,result_LL2y2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2y2y = result_LL2y2y

              !---Integrating LL2y2z
              CALL nag_quad_md_rect_mintg(ay_LL2y2z,a,sing_b,result_LL2y2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2y2z = result_LL2y2z

              !---Integrating LL2z2x
              CALL nag_quad_md_rect_mintg(ay_LL2z2x,a,sing_b,result_LL2z2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2z2x = result_LL2z2x

              !---Integrating LL2z2y
              CALL nag_quad_md_rect_mintg(ay_LL2z2y,a,sing_b,result_LL2z2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2z2y = result_LL2z2y

              !---Integrating LL2z2z
              CALL nag_quad_md_rect_mintg(ay_LL2z2z,a,sing_b,result_LL2z2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2z2z = result_LL2z2z

              !************************************Lower Rectangle**************************
              m_sing_a(1) = xp_surf2(1) - from_sing
              m_sing_b(1) = xp_surf2(1) + from_sing
              m_sing_a(2) = a(2)
              m_sing_b(2) = xp_surf2(2) - from_sing_theta

              !---Integrating UL2x2x
              CAll nag_quad_md_rect_mintg(ay_UL2x2x,m_sing_a,m_sing_b,result_UL2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UL2x2x = sing_result_UL2x2x + result_UL2x2x

              !---Integrating UR2x2x
              CALL nag_quad_md_rect_mintg(ay_UR2x2x,m_sing_a,m_sing_b,result_UR2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UR2x2x = sing_result_UR2x2x + result_UR2x2x

              !---Integrating LR2x2x
              CALL nag_quad_md_rect_mintg(ay_LR2x2x,m_sing_a,m_sing_b,result_LR2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LR2x2x = sing_result_LR2x2x + result_LR2x2x

              !---Integrating LL2x2x
              CALL nag_quad_md_rect_mintg(ay_LL2x2x,m_sing_a,m_sing_b,result_LL2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2x2x = sing_result_LL2x2x + result_LL2x2x

              !---Integrating LL2x2y
              CALL nag_quad_md_rect_mintg(ay_LL2x2y,m_sing_a,m_sing_b,result_LL2x2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2x2y = sing_result_LL2x2y + result_LL2x2y

              !---Integrating LL2x2z
              CALL nag_quad_md_rect_mintg(ay_LL2x2z,m_sing_a,m_sing_b,result_LL2x2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2x2z = sing_result_LL2x2z + result_LL2x2z

              !---Integrating LL2y2x
              CALL nag_quad_md_rect_mintg(ay_LL2y2x,m_sing_a,m_sing_b,result_LL2y2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2y2x = sing_result_LL2y2x + result_LL2y2x

              !---Integrating LL2y2y
              CALL nag_quad_md_rect_mintg(ay_LL2y2y,m_sing_a,m_sing_b,result_LL2y2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2y2y = sing_result_LL2y2y + result_LL2y2y

              !---Integrating LL2y2z
              CALL nag_quad_md_rect_mintg(ay_LL2y2z,m_sing_a,m_sing_b,result_LL2y2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2y2z = sing_result_LL2y2z + result_LL2y2z

              !---Integrating LL2z2x
              CALL nag_quad_md_rect_mintg(ay_LL2z2x,m_sing_a,m_sing_b,result_LL2z2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2z2x = sing_result_LL2z2x + result_LL2z2x

              !---Integrating LL2z2y
              CALL nag_quad_md_rect_mintg(ay_LL2z2y,m_sing_a,m_sing_b,result_LL2z2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2z2y = sing_result_LL2z2y + result_LL2z2y

              !---Integrating LL2z2z
              CALL nag_quad_md_rect_mintg(ay_LL2z2z,m_sing_a,m_sing_b,result_LL2z2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2z2z = sing_result_LL2z2z + result_LL2z2z

              !*****************************************Upper Rectangle***************************************
              m_sing_a(1) = xp_surf2(1) - from_sing
              m_sing_b(1) = xp_surf2(1) + from_sing
              m_sing_a(2) = xp_surf2(2) + from_sing_theta
              m_sing_b(2) = b(2)

              !---Integrating UL2x2x
              CAll nag_quad_md_rect_mintg(ay_UL2x2x,m_sing_a,m_sing_b,result_UL2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UL2x2x = sing_result_UL2x2x + result_UL2x2x

              !---Integrating UR2x2x
              CALL nag_quad_md_rect_mintg(ay_UR2x2x,m_sing_a,m_sing_b,result_UR2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_UR2x2x = sing_result_UR2x2x + result_UR2x2x

              !---Integrating LR2x2x
              CALL nag_quad_md_rect_mintg(ay_LR2x2x,m_sing_a,m_sing_b,result_LR2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LR2x2x = sing_result_LR2x2x + result_LR2x2x

              !---Integrating LL2x2x
              CALL nag_quad_md_rect_mintg(ay_LL2x2x,m_sing_a,m_sing_b,result_LL2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2x2x = sing_result_LL2x2x + result_LL2x2x

              !---Integrating LL2x2y
              CALL nag_quad_md_rect_mintg(ay_LL2x2y,m_sing_a,m_sing_b,result_LL2x2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2x2y = sing_result_LL2x2y + result_LL2x2y

              !---Integrating LL2x2z
              CALL nag_quad_md_rect_mintg(ay_LL2x2z,m_sing_a,m_sing_b,result_LL2x2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2x2z = sing_result_LL2x2z + result_LL2x2z

              !---Integrating LL2y2x
              CALL nag_quad_md_rect_mintg(ay_LL2y2x,m_sing_a,m_sing_b,result_LL2y2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2y2x = sing_result_LL2y2x + result_LL2y2x

              !---Integrating LL2y2y
              CALL nag_quad_md_rect_mintg(ay_LL2y2y,m_sing_a,m_sing_b,result_LL2y2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2y2y = sing_result_LL2y2y + result_LL2y2y

              !---Integrating LL2y2z
              CALL nag_quad_md_rect_mintg(ay_LL2y2z,m_sing_a,m_sing_b,result_LL2y2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2y2z = sing_result_LL2y2z + result_LL2y2z

              !---Integrating LL2z2x
              CALL nag_quad_md_rect_mintg(ay_LL2z2x,m_sing_a,m_sing_b,result_LL2z2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2z2x = sing_result_LL2z2x + result_LL2z2x

              !---Integrating LL2z2y
              CALL nag_quad_md_rect_mintg(ay_LL2z2y,m_sing_a,m_sing_b,result_LL2z2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2z2y = sing_result_LL2z2y + result_LL2z2y

              !---Integrating LL2z2z
              CALL nag_quad_md_rect_mintg(ay_LL2z2z,m_sing_a,m_sing_b,result_LL2z2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              sing_result_LL2z2z = sing_result_LL2z2z + result_LL2z2z

              !***********************Integrating to the Right of the Singularity**************************
              sing_a(1) = xp_surf2(1) + from_sing
              sing_a(2) = a(2)
              
              !---Integrating UL2x2x
              CALL nag_quad_md_rect_mintg(ay_UL2x2x,sing_a,b,result_UL2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_UL2x2x = sing_result_UL2x2x + result_UL2x2x
              data_UL2x2x(i_mid_pts,column_index) = cmplx(result_UL2x2x(1)+.5_wp,result_UL2x2x(2),wp)

              !---Integrating UR2x2x
              CALL nag_quad_md_rect_mintg(ay_UR2x2x,sing_a,b,result_UR2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_UR2x2x = sing_result_UR2x2x + result_UR2x2x
              data_UR2x2x(i_mid_pts,column_index) = cmplx(result_UR2x2x(1),result_UR2x2x(2),wp)

              !---Integrating LR2x2x
              CALL nag_quad_md_rect_mintg(ay_LR2x2x,sing_a,b,result_LR2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LR2x2x = sing_result_LR2x2x + result_LR2x2x
              data_LR2x2x(i_mid_pts,column_index) = cmplx(result_LR2x2x(1),result_LR2x2x(2),wp)

              !---Integraing LL2x2x
              CALL nag_quad_md_rect_mintg(ay_LL2x2x,sing_a,b,result_LL2x2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL2x2x = sing_result_LL2x2x + result_LL2x2x !c11
              data_LL2x2x(i_mid_pts,column_index) = cmplx(result_LL2x2x(1),result_LL2x2x(2),wp)-.5_wp*c11_2(x(1),x(2))

              !---Integraing LL2x2y
              CALL nag_quad_md_rect_mintg(ay_LL2x2y,sing_a,b,result_LL2x2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL2x2y = sing_result_LL2x2y + result_LL2x2y !c12
              data_LL2x2y(i_mid_pts,column_index) = cmplx(result_LL2x2y(1),result_LL2x2y(2),wp)-.5_wp*c12_2(x(1),x(2))

              !---Integraing LL2x2z
              CALL nag_quad_md_rect_mintg(ay_LL2x2z,sing_a,b,result_LL2x2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL2x2z = sing_result_LL2x2z + result_LL2x2z !c13
              data_LL2x2z(i_mid_pts,column_index) = cmplx(result_LL2x2z(1),result_LL2x2z(2),wp)-.5_wp*c13_2(x(1),x(2))

              !---Integraing LL2y2x
              CALL nag_quad_md_rect_mintg(ay_LL2y2x,sing_a,b,result_LL2y2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL2y2x = sing_result_LL2y2x + result_LL2y2x !c12=c21
              data_LL2y2x(i_mid_pts,column_index) = cmplx(result_LL2y2x(1),result_LL2y2x(2),wp)-.5_wp*c12_2(x(1),x(2))

              !---Integraing LL2y2y
              CALL nag_quad_md_rect_mintg(ay_LL2y2y,sing_a,b,result_LL2y2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL2y2y = sing_result_LL2y2y + result_LL2y2y !c22
              data_LL2y2y(i_mid_pts,column_index) = cmplx(result_LL2y2y(1),result_LL2y2y(2),wp)-.5_wp*c22_2(x(1),x(2))

              !---Integraing LL2y2z
              CALL nag_quad_md_rect_mintg(ay_LL2y2z,sing_a,b,result_LL2y2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL2y2z = sing_result_LL2y2z + result_LL2y2z !c23
              data_LL2y2z(i_mid_pts,column_index) = cmplx(result_LL2y2z(1),result_LL2y2z(2),wp)-.5_wp*c23_2(x(1),x(2))

              !---Integraing LL2z2x
              CALL nag_quad_md_rect_mintg(ay_LL2z2x,sing_a,b,result_LL2z2x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL2z2x = sing_result_LL2z2x + result_LL2z2x  !c13=c31
              data_LL2z2x(i_mid_pts,column_index) = cmplx(result_LL2z2x(1),result_LL2z2x(2),wp)-.5_wp*c13_2(x(1),x(2))

              !---Integraing LL2z2y
              CALL nag_quad_md_rect_mintg(ay_LL2z2y,sing_a,b,result_LL2z2y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL2z2y = sing_result_LL2z2y + result_LL2z2y  !c23=c32
              data_LL2z2y(i_mid_pts,column_index) = cmplx(result_LL2z2y(1),result_LL2z2y(2),wp)-.5_wp*c23_2(x(1),x(2))

              !---Integraing LL2z2z
              CALL nag_quad_md_rect_mintg(ay_LL2z2z,sing_a,b,result_LL2z2z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, &
                   abs_acc=abs_acc)
              result_LL2z2z = sing_result_LL2z2z + result_LL2z2z !c33
              data_LL2z2z(i_mid_pts,column_index) = cmplx(result_LL2z2z(1),result_LL2z2z(2),wp)-.5_wp*c33_2(x(1),x(2))

           end if

           !---Integrating UL2x1x
           CALL nag_quad_md_rect_mintg(ay_UL2x1x,a,b,result_UL2x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_UL2x1x(i_mid_pts,column_index) = cmplx(result_UL2x1x(1),result_UL2x1x(2),wp)

           !---Integrating UR2x1x
           CALL nag_quad_md_rect_mintg(ay_UR2x1x,a,b,result_UR2x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_UR2x1x(i_mid_pts,column_index) = cmplx(result_UR2x1x(1),result_UR2x1x(2),wp)

           !---Integrating LR2x1x
           CALL nag_quad_md_rect_mintg(ay_LR2x1x,a,b,result_LR2x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LR2x1x(i_mid_pts,column_index) = cmplx(result_LR2x1x(1),result_LR2x1x(2),wp)

           !---Integrating LL2x1x
           CALL nag_quad_md_rect_mintg(ay_LL2x1x,a,b,result_LL2x1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL2x1x(i_mid_pts,column_index) = cmplx(result_LL2x1x(1),result_LL2x1x(2),wp)

           !---Integrating LL2x1y
           CALL nag_quad_md_rect_mintg(ay_LL2x1y,a,b,result_LL2x1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL2x1y(i_mid_pts,column_index) = cmplx(result_LL2x1y(1),result_LL2x1y(2),wp)

           !---Integrating LL2x1z
           CALL nag_quad_md_rect_mintg(ay_LL2x1z,a,b,result_LL2x1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL2x1z(i_mid_pts,column_index) = cmplx(result_LL2x1z(1),result_LL2x1z(2),wp)

           !---Integrating LL2y1x
           CALL nag_quad_md_rect_mintg(ay_LL2y1x,a,b,result_LL2y1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL2y1x(i_mid_pts,column_index) = cmplx(result_LL2y1x(1),result_LL2y1x(2),wp)

           !---Integrating LL2y1y
           CALL nag_quad_md_rect_mintg(ay_LL2y1y,a,b,result_LL2y1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL2y1y(i_mid_pts,column_index) = cmplx(result_LL2y1y(1),result_LL2y1y(2),wp)

           !---Integrating LL2y1z
           CALL nag_quad_md_rect_mintg(ay_LL2y1z,a,b,result_LL2y1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL2y1z(i_mid_pts,column_index) = cmplx(result_LL2y1z(1),result_LL2y1z(2),wp)

           !---Integrating LL2z1x
           CALL nag_quad_md_rect_mintg(ay_LL2z1x,a,b,result_LL2z1x, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL2z1x(i_mid_pts,column_index) = cmplx(result_LL2z1x(1),result_LL2z1x(2),wp)

           !---Integrating LL2z1y
           CALL nag_quad_md_rect_mintg(ay_LL2z1y,a,b,result_LL2z1y, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL2z1y(i_mid_pts,column_index) = cmplx(result_LL2z1y(1),result_LL2z1y(2),wp)

           !---Integrating LL2z1z
           CALL nag_quad_md_rect_mintg(ay_LL2z1z,a,b,result_LL2z1z, max_fun_eval=max_fun_eval, rel_acc=rel_acc, abs_acc=abs_acc)
           data_LL2z1z(i_mid_pts,column_index) = cmplx(result_LL2z1z(1),result_LL2z1z(2),wp)
          
           column_index = column_index+1 
 
        end do
        DEALLOCATE(theta_p)

        if( b(1)>ay_R ) then !b(1) must be <= ay_R.  Due to round off errors b(1) may become larger than ay_R
           write(*,*) 'b(1)>ay_R, Error for Surface 2, aborting'
           stop
        end if

     end do !ending i_rho_p loop

     DEALLOCATE(rho_p)

  end do !endding i_mid_pts loop

  DEALLOCATE(x_mid_pts_S2) !Deallocating Memory
END SUBROUTINE AY_MATRIX_FILL

SUBROUTINE ay_Einc(Einc)
  IMPLICIT NONE
  COMPLEX(wp), INTENT(OUT) :: Einc(:)
  !LOCAL
  INTEGER :: num_sample_pts, i
  REAL(wp), ALLOCATABLE :: x_mid_pts_S1(:,:)
  REAL(wp), ALLOCATABLE :: x_mid_pts_S2(:,:)

  num_sample_pts = ay_tot_num_obs_pts() !Determines total number of obs pts/Surf
  Einc(1:6*num_sample_pts) = (0.0_wp,0.0_wp)

  ALLOCATE( x_mid_pts_S1(num_sample_pts,3) )
  ALLOCATE( x_mid_pts_S2(num_sample_pts,3) )
  CALL ay_obs_pts(x_mid_pts_S1,1) 
  CALL ay_obs_pts(x_mid_pts_S2,2)

  do i=1, num_sample_pts
     Einc(i) = 1.0_wp*exp(ay_I*ay_ko*x_mid_pts_S1(i,3))
     Einc(3*num_sample_pts+i) = 1.0_wp*exp(ay_I*ay_ko*x_mid_pts_S2(i,3))
  end do
  
  DEALLOCATE(x_mid_pts_S1)
  DEALLOCATE(x_mid_pts_S2)
  
END SUBROUTINE ay_Einc

SUBROUTINE ay_Einc_norm(Einc_norm)
  IMPLICIT NONE
  COMPLEX(wp), INTENT(OUT) :: Einc_norm(:)
  !LOCAL
  INTEGER :: num_sample_pts, i
  REAL(wp), ALLOCATABLE :: x_mid_pts_S1(:,:)
  REAL(wp), ALLOCATABLE :: x_mid_pts_S2(:,:)

  num_sample_pts = ay_tot_num_obs_pts() !Determines total number of obs pts/Surf
  Einc_norm(1:6*num_sample_pts) = (0.0_wp,0.0_wp)

  ALLOCATE( x_mid_pts_S1(num_sample_pts,3) )
  ALLOCATE( x_mid_pts_S2(num_sample_pts,3) )
  CALL ay_obs_pts(x_mid_pts_S1,1)
  CALL ay_obs_pts(x_mid_pts_S2,2)

  do i=1, num_sample_pts
     Einc_norm(i) = 1.0_wp*ay_I*ay_ko*exp(ay_I*ay_ko*x_mid_pts_S1(i,3)) 
     Einc_norm(3*num_sample_pts+i) = -1.0_wp*ay_I*ay_ko*exp(ay_I*ay_ko*x_mid_pts_S2(i,3)) 
  end do

  DEALLOCATE(x_mid_pts_S1)
  DEALLOCATE(x_mid_pts_S2)

END SUBROUTINE ay_Einc_norm

END MODULE AY_MATRIX
